# Fine-tuning Granite Vision 3.1 2B with TRL

_Authored by: [Eli Schwartz](https://huggingface.co/elischwartz)_

Adapted from [Sergio Paniego](https://github.com/sergiopaniego)'s [Notebook](https://huggingface.co/learn/cookbook/en/fine_tuning_smol_vlm_sft_trl)


This recipe will enable you to fine-tune [IBM's Granite Vision 3.1 2B Model](https://huggingface.co/ibm-granite/granite-vision-3.1-2b-preview).
It is a lightweight yet capable model trained by fine-tuning a [Granite language model](https://huggingface.co/ibm-granite/granite-3.1-2b-instruct) with both image and text modalities.
We will be using the Hugging Face ecosystem, leveraging the powerful [Transformer Reinforcement Learning library (TRL)](https://huggingface.co/docs/trl/index). This step-by-step guide will enable you to Granite Vision for your specific tasks, even on consumer GPUs.

### 🌟 Model & Dataset Overview

In this notebook, we will fine-tune and evaluate the **[Granite Vision](https://huggingface.co/ibm-granite/granite-vision-3.1-2b-preview)** model using the **[Geometric Perception](https://huggingface.co/datasets/euclid-multimodal/Geoperception)** dataset, containing tasks that the model wasn't intially trained for. Granite Vision is a highly performant and memory-efficient model, making it an ideal for fine tuning for new tasks. The **Geometric Perception** provides images of various geometric diagrams, compiled from high-school textbooks, paired with question-answer pairs.


This notebook is tested using a A100 GPU.

## 1. Install Dependencies

Let’s start by installing the essential libraries we’ll need for fine-tuning! 🚀


```python
!pip install -q git+https://github.com/huggingface/transformers.git
!pip install  -U -q trl datasets bitsandbytes peft accelerate
# Tested with transformers==4.49.0.dev0, trl==0.14.0, datasets==3.2.0, bitsandbytes==0.45.2, peft==0.14.0, accelerate==1.3.0
```

```python
>>> !pip install -q flash-attn --no-build-isolation

>>> try:
...     import flash_attn
...     print("FlashAttention is installed")
...     USE_FLASH_ATTENTION = True
>>> except ImportError:
...     print("FlashAttention is not installed")
...     USE_FLASH_ATTENTION = False
```

<pre>
FlashAttention is not installed
</pre>

## 2. Load Dataset 📁

We’ll load the **[Geometric Perception](https://huggingface.co/datasets/euclid-multimodal/Geoperception)** dataset, which provides images of various geometric diagrams, compiled from popular high-school textbooks, paired with question-answer pairs.

We’ll use the original system prompt used during the model training.

```python
system_message = "A chat between a curious user and an artificial intelligence assistant. The assistant gives helpful, detailed, and polite answers to the user's questions."
```

For educational purposes, we’ll only train and evaluate on the Line Length Comaprison task, specified in the "predicate" field of the dataset.

```python
from datasets import load_dataset

dataset_id = "euclid-multimodal/Geoperception"
dataset = load_dataset(dataset_id)
dataset_LineComparison = dataset['train'].filter(lambda x: x['predicate'] == 'LineComparison')
train_test = dataset_LineComparison.train_test_split(test_size=0.5, seed=42)
```

Let’s take a look at the dataset structure. It includes an image, a question, an answer, and "predicate" which we used to filter the dataset.

```python
train_test
```

We’ll format the dataset into a chatbot structure, with the system message, image, user query, and answer for each interaction.

💡For more tips on using this model for inference, check out the [Model Card](https://huggingface.co/ibm-granite/granite-vision-3.1-2b-preview).

```python
def format_data(sample):
    return [
        {
            "role": "system",
            "content": [
                {
                    "type": "text",
                    "text": system_message
                }
            ],
        },
        {
            "role": "user",
            "content": [
                {
                    "type": "image",
                    "image": sample["image"],
                },
                {
                    "type": "text",
                    "text": sample['question'],
                }
            ],
        },
        {
            "role": "assistant",
            "content": [
                {
                    "type": "text",
                    "text": sample["answer"]
                }
            ],
        },
    ]
```

Now, let’s format the data using the chatbot structure. This will set up the interactions for the model.

```python
train_dataset = [format_data(x) for x in train_test['train']]
test_dataset = [format_data(x) for x in train_test['test']]
```

```python
train_dataset[200]
```

## 3. Load Model and Check Performance! 🤔

Now that we’ve loaded the dataset, it’s time to load the [IBM's Granite Vision Model](https://huggingface.co/ibm-granite/granite-vision-3.1-2b-preview), a 2B parameter Vision Language Model (VLM) built on that offers state-of-the-art (SOTA) performance while being efficient in terms of memory usage.

For a broader comparison of state-of-the-art VLMs, explore the [WildVision Arena](https://huggingface.co/spaces/WildVision/vision-arena) and the [OpenVLM Leaderboard](https://huggingface.co/spaces/opencompass/open_vlm_leaderboard), where you can find the best-performing models across various benchmarks.


```python
import torch
from transformers import AutoModelForVision2Seq, AutoProcessor

model_id = "ibm-granite/granite-vision-3.1-2b-preview"
```

Next, we’ll load the model and the tokenizer to prepare for inference.

```python
model = AutoModelForVision2Seq.from_pretrained(
    model_id,
    device_map="auto",
    torch_dtype=torch.bfloat16,
    _attn_implementation="flash_attention_2" if USE_FLASH_ATTENTION else None,
)

processor = AutoProcessor.from_pretrained(model_id)
```

To evaluate the model's performance, we’ll use a sample from the dataset. First, let’s inspect the internal structure of this sample to understand how the data is organized.

```python
test_idx = 20
sample = test_dataset[test_idx]
sample
```

Now, let’s take a look at the image corresponding to the sample. Can you answer the query based on the visual information?


```python
>>> sample[1]['content'][0]['image']
```

<img src="data:image/jpeg;base64,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">


Let’s create a method that takes the model, processor, and sample as inputs to generate the model's answer. This will allow us to streamline the inference process and easily evaluate the VLM's performance.

```python
def generate_text_from_sample(model, processor, sample, max_new_tokens=100, device="cuda"):
    # Prepare the text input by applying the chat template
    text_input = processor.apply_chat_template(
        sample[:2],  # Use the sample without the assistant response
        add_generation_prompt=True
    )

    image_inputs = []
    image = sample[1]['content'][0]['image']
    if image.mode != 'RGB':
        image = image.convert('RGB')
    image_inputs.append([image])

    # Prepare the inputs for the model
    model_inputs = processor(
        #text=[text_input],
        text=text_input,
        images=image_inputs,
        return_tensors="pt",
    ).to(device)  # Move inputs to the specified device

    # Generate text with the model
    generated_ids = model.generate(**model_inputs, max_new_tokens=max_new_tokens)

    # Trim the generated ids to remove the input ids
    trimmed_generated_ids = [
        out_ids[len(in_ids):] for in_ids, out_ids in zip(model_inputs.input_ids, generated_ids)
    ]

    # Decode the output text
    output_text = processor.batch_decode(
        trimmed_generated_ids,
        skip_special_tokens=True,
        clean_up_tokenization_spaces=False
    )

    return output_text[0]  # Return the first decoded output text
```

```python
output = generate_text_from_sample(model, processor, sample)
output
```

It seems like the model is unable to comapre the lines' lengths which are not explicitly specified. To improve its performance, we can fine-tune the model with more relevant data to ensure it better understands the context and provides more accurate responses.

**Remove Model and Clean GPU**

Before we proceed with training the model in the next section, let's clear the current variables and clean the GPU to free up resources.



```python
>>> import gc
>>> import time

>>> def clear_memory():
...     # Delete variables if they exist in the current global scope
...     if 'inputs' in globals(): del globals()['inputs']
...     if 'model' in globals(): del globals()['model']
...     if 'processor' in globals(): del globals()['processor']
...     if 'trainer' in globals(): del globals()['trainer']
...     if 'peft_model' in globals(): del globals()['peft_model']
...     if 'bnb_config' in globals(): del globals()['bnb_config']
...     time.sleep(2)

...     # Garbage collection and clearing CUDA memory
...     gc.collect()
...     time.sleep(2)
...     torch.cuda.empty_cache()
...     torch.cuda.synchronize()
...     time.sleep(2)
...     gc.collect()
...     time.sleep(2)

...     print(f"GPU allocated memory: {torch.cuda.memory_allocated() / 1024**3:.2f} GB")
...     print(f"GPU reserved memory: {torch.cuda.memory_reserved() / 1024**3:.2f} GB")

>>> clear_memory()
```

<pre>
GPU allocated memory: 0.01 GB
GPU reserved memory: 0.02 GB
</pre>

## 4. Fine-Tune the Model using TRL


### 4.1 Load the Quantized Model for Training ⚙️

Next, we’ll load the quantized model using [bitsandbytes](https://huggingface.co/docs/bitsandbytes/main/en/index). If you want to learn more about quantization, check out [this blog post](https://huggingface.co/blog/merve/quantization) or [this one](https://www.maartengrootendorst.com/blog/quantization/).


```python
from transformers import BitsAndBytesConfig

USE_QLORA = True
USE_LORA = True

if USE_QLORA:
    # BitsAndBytesConfig int-4 config
    bnb_config = BitsAndBytesConfig(
        load_in_4bit=True,
        bnb_4bit_use_double_quant=True,
        bnb_4bit_quant_type="nf4",
        bnb_4bit_compute_dtype=torch.bfloat16,
        llm_int8_skip_modules=["vision_tower", "lm_head"],  # Skip problematic modules
        llm_int8_enable_fp32_cpu_offload=True
    )
else:
    bnb_config = None

# Load model and tokenizer
model = AutoModelForVision2Seq.from_pretrained(
    model_id,
    device_map="auto",
    torch_dtype=torch.bfloat16,
    quantization_config=bnb_config,
    _attn_implementation="flash_attention_2" if USE_FLASH_ATTENTION else None,
)
processor = AutoProcessor.from_pretrained(model_id)
```

### 4.2 Set Up QLoRA and SFTConfig 🚀

Next, we’ll configure [QLoRA](https://github.com/artidoro/qlora) for our training setup. QLoRA allows efficient fine-tuning of large models by reducing the memory footprint. Unlike traditional LoRA, which uses low-rank approximation, QLoRA further quantizes the LoRA adapter weights, leading to even lower memory usage and faster training.

To boost efficiency, we can also leverage a **paged optimizer** or **8-bit optimizer** during QLoRA implementation. This approach enhances memory efficiency and speeds up computations, making it ideal for optimizing our model without sacrificing performance.

```python
if USE_LORA:
    from peft import LoraConfig, get_peft_model
    
    # Configure LoRA
    peft_config = LoraConfig(
        r=8,
        lora_alpha=8,
        lora_dropout=0.1,
        target_modules=[name for name, _ in model.named_modules() if 'language_model' in name and '_proj' in name],
        use_dora=True,
        init_lora_weights="gaussian"
    )
    
    # Apply PEFT model adaptation
    # model = get_peft_model(model, peft_config)
    model.add_adapter(peft_config)
    model.enable_adapters()
    model = get_peft_model(model, peft_config)
    
    # Print trainable parameters
    model.print_trainable_parameters()
    
else:
    peft_config = None
```

We will use Supervised Fine-Tuning (SFT) to improve our model's performance on the specific task. To achieve this, we'll define the training arguments with the [SFTConfig](https://huggingface.co/docs/trl/sft_trainer) class from the [TRL library](https://huggingface.co/docs/trl/index). SFT leverages labeled data to help the model generate more accurate responses, adapting it to the task. This approach enhances the model's ability to understand and respond to visual queries more effectively.

```python
from trl import SFTConfig

# Configure training arguments using SFTConfig
training_args = SFTConfig(
    output_dir="./checkpoints/geoperception",
    num_train_epochs=1,
    # max_steps=30,
    per_device_train_batch_size=8,
    gradient_accumulation_steps=2,
    warmup_steps=10,
    learning_rate=1e-4,
    weight_decay=0.01,
    logging_steps=10,
    save_strategy="steps",
    save_steps=20,
    save_total_limit=1,
    optim="adamw_torch_fused",
    bf16=True,
    push_to_hub=False,
    report_to="none",
    remove_unused_columns=False,
    gradient_checkpointing=True,
    dataset_text_field="",
    dataset_kwargs={"skip_prepare_dataset": True},
)
```

### 4.3 Training the Model 🏃

To ensure that the data is correctly structured for the model during training, we need to define a collator function. This function will handle the formatting and batching of our dataset inputs, ensuring the data is properly aligned for training.

👉 For more details, check out the official [TRL example scripts](https://github.com/huggingface/trl/blob/main/examples/scripts/sft_vlm_smol_vlm.py).

```python
def collate_fn(examples):
    texts = [processor.apply_chat_template(example, tokenize=False) for example in examples]

    image_inputs = []
    for example in examples:
      image = example[1]['content'][0]['image']
      if image.mode != 'RGB':
          image = image.convert('RGB')
      image_inputs.append([image])

    batch = processor(text=texts, images=image_inputs, return_tensors="pt", padding=True)

    labels = batch["input_ids"].clone()
    assistant_tokens = processor.tokenizer("<|assistant|>", return_tensors="pt")['input_ids'][0]
    eos_token = processor.tokenizer("<|end_of_text|>", return_tensors="pt")['input_ids'][0]

    for i in range(batch["input_ids"].shape[0]):
        apply_loss = False
        for j in range(batch["input_ids"].shape[1]):
            if not apply_loss:
                labels[i][j] = -100
            if ((j>=len(assistant_tokens)+1) and
                torch.all(batch["input_ids"][i][j+1-len(assistant_tokens):j+1]==assistant_tokens)):
                apply_loss = True
            if batch["input_ids"][i][j]==eos_token:
                apply_loss = False

    batch["labels"] = labels

    return batch
```

Now, we will define the [SFTTrainer](https://huggingface.co/docs/trl/sft_trainer), which is a wrapper around the [transformers.Trainer](https://huggingface.co/docs/transformers/main_classes/trainer) class and inherits its attributes and methods. This class simplifies the fine-tuning process by properly initializing the [PeftModel](https://huggingface.co/docs/peft/v0.6.0/package_reference/peft_model) when a [PeftConfig](https://huggingface.co/docs/peft/v0.6.0/en/package_reference/config#peft.PeftConfig) object is provided. By using `SFTTrainer`, we can efficiently manage the training workflow and ensure a smooth fine-tuning experience for our Vision Language Model.



```python
from trl import SFTTrainer

trainer = SFTTrainer(
    model=model,
    args=training_args,
    train_dataset=train_dataset,
    data_collator=collate_fn,
    peft_config=peft_config,
    processing_class=processor.tokenizer,
)
```

Time to Train the Model! 🎉

```python
trainer.train()
```

Let's save the results 💾

```python
trainer.save_model(training_args.output_dir)
```

## 5. Testing the Fine-Tuned Model 🔍

Now that our Vision Language Model (VLM) is fine-tuned, it's time to evaluate its performance! In this section, we'll test the model using examples from the ChartQA dataset to assess how accurately it answers questions based on chart images. Let's dive into the results and see how well it performs! 🚀

Let's clean up the GPU memory to ensure optimal performance 🧹

```python
>>> clear_memory()
```

<pre>
GPU allocated memory: 0.02 GB
GPU reserved memory: 0.19 GB
</pre>

We will reload the base model using the same pipeline as before.

```python
model = AutoModelForVision2Seq.from_pretrained(
    training_args.output_dir,
    device_map="auto",
    torch_dtype=torch.bfloat16,
    _attn_implementation="flash_attention_2" if USE_FLASH_ATTENTION else None,
)

processor = AutoProcessor.from_pretrained(model_id)
```

We will merge the LORA adapters in case we are using them.

```python
if USE_LORA:
    from peft import PeftModel
    model = PeftModel.from_pretrained(model, training_args.output_dir)
```

Let's evaluate the model on an unseen sample.


```python
test_idx = 20
sample = test_dataset[test_idx]
sample[1:]
```

```python
>>> sample[1]['content'][0]['image']
```

<img src="data:image/jpeg;base64,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">


```python
output = generate_text_from_sample(model, processor, sample)
output
```

#### 🎉✨ The model has successfully learned to respond to the queries as specified in the dataset. We've achieved our goal! 🎉✨

<EditOnGithub source="https://github.com/huggingface/cookbook/blob/main/notebooks/en/fine_tuning_granite_vision_sft_trl.md" />