# Hyperparameter Optimization with Optuna and Transformers

_Authored by: [Parag Ekbote](https://github.com/ParagEkbote)_

## **Problem:** 
Find the best hyperparameters to fine-tune a lightweight BERT model for text classification on a subset of the IMDB dataset.

## **Overview:**
This recipe demonstrates how to systematically optimize hyperparameters for transformer-based text classification models using automated search techniques. You'll learn to implement HPO using Optuna to find optimal learning rates and weight decay values for fine-tuning BERT on sentiment analysis tasks.

## **When to Use This Recipe:**

* You need to fine-tune pre-trained language models for classification tasks.

* Your model performance is plateauing and requires parameter refinement.

* You want to implement systematic, reproducible hyperparameter optimization.

### Notes

* For detailed guidance on hyperparameter search with Transformers, refer to the [Hugging Face HPO documentation](https://huggingface.co/docs/transformers/en/hpo_train).

```python
!pip install -q datasets evaluate transformers optuna wandb scikit-learn nbformat matplotlib
```

## Prepare Dataset and Set Model

Before you can train and evaluate a sentiment analysis model, you’ll need to prep the dataset. This section ensures that your data is structured and your model is primed for learning from scratch or fine-tuning in the case of BERT.


### 1. **Load the IMDB Dataset**  
   Begin by selecting a dataset focused on sentiment classification. IMDB is a well-known benchmark that features movie reviews labeled as either positive or negative.

### 2. **Select Input and Output Columns**  
   Focus only on the essentials:  
   - `text` column serves as the input (review content)  
   - `label` column serves as the target (0 for negative, 1 for positive sentiment)

### 3. **Define the Train/Validation Split**  
   Choose a consistent sampling strategy by selecting:  
   - 2000 examples for training  
   - 1000 examples for validation  
   Use a fixed random seed when shuffling to ensure reproducibility across sessions.

### 4. **Tokenize the Dataset**  
   Apply a tokenizer compatible with the model you're planning to use. Tokenization converts raw text into numerical format so the model can ingest it effectively. Use batch processing to make this step efficient.

### 5. **Load an Evaluation Metric**  
   Choose “accuracy” as the primary evaluation metric—simple and effective for binary classification tasks like this. It will later help gauge how well your model is learning the difference between positive and negative sentiment.

### 6. **Initialize a Pretrained BERT Model**  
   Select a pretrained BERT-based model tailored for sequence classification tasks. Set the number of output classes to 2 (positive and negative) to align with your sentiment labels. This model will serve as the learner throughout the training process.

```python
from datasets import load_dataset
import evaluate

from transformers import AutoModelForSequenceClassification
from transformers import AutoTokenizer
from transformers import set_seed

set_seed(42)


train_dataset = load_dataset("imdb", split="train").shuffle(seed=42).select(range(2500))
valid_dataset = load_dataset("imdb", split="test").shuffle(seed=42).select(range(1000))

model_name = "prajjwal1/bert-tiny"
tokenizer = AutoTokenizer.from_pretrained(model_name)

def tokenize(batch):
    return tokenizer(batch["text"], padding="max_length", truncation=True, max_length=512)

tokenized_train = train_dataset.map(tokenize, batched=True).select_columns(
    ["input_ids", "attention_mask", "label"]
)
tokenized_valid = valid_dataset.map(tokenize, batched=True).select_columns(
    ["input_ids", "attention_mask", "label"]
)

metric = evaluate.load("accuracy")


def model_init():
    return AutoModelForSequenceClassification.from_pretrained(model_name, num_labels=2)
```

## Define Storage with Optuna

To ensure your hyperparameter optimization experiments are trackable, reproducible, and easy to analyze over time, it’s essential to use a persistent storage backend. Optuna provides a robust solution for this through its `RDBStorage` mechanism, which allows saving trial data across multiple sessions using an SQLite database.


### 1. **Choose a Persistent Storage Format**  
   Opt for an SQLite database as the storage medium. It’s lightweight, portable, and ideal for local experimentation, while still enabling structured access to all trial data.

### 2. **Enable Optuna's RDBStorage**  
   RDBStorage (Relational Database Storage) is Optuna’s way of saving trial results in a consistent and queryable format. This bridges the gap between short-term experimentation and long-term analysis.

### 3. **Preserve Trial History Across Sessions**  
   By setting up persistent storage, you ensure that every hyperparameter trial is recorded. You can pause and resume studies, add more trials later, or analyze outcomes long after training has ended.

### 4. **Facilitate Reproducible Analysis**  
   With trials stored centrally, you can revisit earlier results, regenerate visualizations, or compare different optimization runs. This makes your workflow transparent, collaborative, and scientifically rigorous.

### 5. **Support Visualization and Monitoring Tools**  
   Storing trials persistently lets you plug in visualization tools—like Optuna’s built-in plotting utilities or external dashboards—to inspect performance trends and refine your search space iteratively.

```python
import optuna
from optuna.storages import RDBStorage

# Define persistent storage
storage = RDBStorage("sqlite:///optuna_trials.db")

study = optuna.create_study(
    study_name="transformers_optuna_study",
    direction="maximize",
    storage=storage,
    load_if_exists=True
)
```

## Initialize Trainer and Set Up Observability

Now that your hyperparameter search space is in place, the next step is to wire everything together for optimization and tracking. This setup ensures not only accuracy-driven tuning but also full visibility into the training process using Weight & Biases (W&B).

### 1. **Define the Metric Function**  
   Start by specifying how model performance will be measured. This metric function evaluates predictions after each validation step to calculate accuracy, F1-score, or loss. It becomes the feedback loop guiding each trial’s learning progress.

### 2. **Construct the Objective Function**  
   This is the centerpiece of hyperparameter optimization. It wraps your training loop and returns a scalar score based on the chosen metric (like validation accuracy). Optuna will use this to compare trials and decide which settings yield the best outcomes.

### 3. **Set Up Weight & Biases for Observability**  
   Configure your environment to log experiment metrics and hyperparameter configurations to W&B. This platform offers dashboards, plots, and experiment comparisons to track progress and spot issues.

### 4. **Authenticate for Logging Access**  
   Log in to W&B using your personal API key. This step connects your training session to your online account so that all metrics and trial details are properly tracked and stored.

### 5. **Define Trainer Arguments**  
   Prepare a configuration setup for your training manager (such as the Hugging Face `Trainer`). Include settings for:
   - Evaluation strategy (e.g. after every epoch)
   - Checkpoint frequency and save conditions
   - Logging intervals
   - Hyperparameter search method and objectives

   This ensures that training is robust, reproducible, and easy to resume or analyze.

```python
import wandb
from transformers import Trainer, TrainingArguments

def compute_metrics(eval_pred):
    predictions = eval_pred.predictions.argmax(axis=-1)
    labels = eval_pred.label_ids
    return metric.compute(predictions=predictions, references=labels)


def compute_objective(metrics):
    return metrics["eval_accuracy"]

wandb.init(project="hf-optuna", name="transformers_optuna_study")

training_args = TrainingArguments(
    output_dir="./results",
        eval_strategy="epoch",
        save_strategy="epoch",
        load_best_model_at_end=True,
        logging_strategy="epoch",
        num_train_epochs=3,
        report_to="wandb",  # Logs to W&B
        logging_dir="./logs",
        run_name="transformers_optuna_study",
)


trainer = Trainer(
    model_init=model_init,
    args=training_args,
    train_dataset=tokenized_train,
    eval_dataset=tokenized_valid,
    processing_class=tokenizer,
    compute_metrics=compute_metrics,
)
```

## Define Search Space and Start Trials

Before diving into model training, it’s essential to thoughtfully define the ingredients and the exploration strategy. This step sets the stage for hyperparameter optimization using Optuna, where you'll systematically explore combinations of training parameters like learning rate, weight decay, and batch size.

### 1. **Design the Search Space**  
   Begin by outlining the hyperparameters you want to optimize. Choose reasonable lower and upper bounds for each:
   - *Learning rate*—controls the step size during optimization.
   - *Weight decay*—adds regularization to reduce overfitting.
   - *Batch size*—affects memory use and convergence stability.

### 2. **Set the Optimization Direction**  
   Decide whether your goal is to minimize (e.g. loss) or maximize (e.g. accuracy, F1 score) the evaluation metric. This guides the search engine in the right direction.

### 3. **Choose Optuna as the Backend**  
   Optuna will handle the search process—selecting, evaluating, and iterating through hyperparameter combinations intelligently.

### 4. **Specify the Number of Trials**  
   Define how many individual runs ("trials") you want Optuna to attempt. More trials can explore the space better but take more time.

### 5. **Define the Objective Function**  
   This function calculates the metric to be optimized during each trial. It encapsulates how the model is trained and how performance is evaluated after each configuration is tested.

### 6. **Name the Study for Persistence**  
   Assign a name to your study so it can be resumed or referenced later. This is especially useful when running experiments over multiple sessions or machines.

### 7. **Set Up Persistent Storage**  
   Choose the storage backend that we previously setup to let you continue the study later, analyze results, or visualize metrics even after a system reboot.


```python
>>> def optuna_hp_space(trial):
...     return {
...         "learning_rate": trial.suggest_float("learning_rate", 1e-6, 1e-4, log=True),
...         "per_device_train_batch_size": trial.suggest_categorical(
...             "per_device_train_batch_size", [16, 32, 64, 128]
...         ),
...         "weight_decay": trial.suggest_float("weight_decay", 0.0, 0.3),
...     }


>>> best_run = trainer.hyperparameter_search(
...     direction="maximize",
...     backend="optuna",
...     hp_space=optuna_hp_space,
...     n_trials=5,
...     compute_objective=compute_objective,
...     study_name="transformers_optuna_study",
...     storage="sqlite:///optuna_trials.db",
...     load_if_exists=True
... )

>>> print(best_run)
```

<pre>
BestRun(run_id='0', objective=0.764, hyperparameters={'learning_rate': 7.23655165533393e-05, 'per_device_train_batch_size': 16, 'weight_decay': 0.013798094328723032}, run_summary=None)
</pre>

## Visualize Results

Once your Optuna study completes its trials, it’s time to peel back the layers and interpret what happened. Visualization brings clarity to how hyperparameters shaped the outcome and uncovers patterns that might otherwise stay buried in raw data.

### 1. **Track Optimization Progress**  
   Use the optimization history to see how objective scores evolved over trials. This helps you understand whether performance steadily improved, plateaued, or oscillated. It’s your window into the pace and trajectory of the search process.

### 2. **Inspect Training Behavior via Intermediate Values**  
   If your model reports evaluation metrics during training (like per epoch), intermediate value plots let you monitor how each trial performed in real time. This is especially valuable for early-stopping decisions and assessing learning stability.

### 3. **Reveal Key Hyperparameters through Importance Rankings**  
   Parameter importance plots uncover which hyperparameters actually mattered—did tweaking the learning rate move the needle, or was batch size the star? Understanding this lets you simplify or refine your future search space.

```python
>>> import optuna
>>> from optuna.visualization.matplotlib import (
...     plot_optimization_history,
...     plot_intermediate_values,
...     plot_param_importances
... )
>>> import matplotlib.pyplot as plt

>>> # Load the study from RDB storage
>>> storage = optuna.storages.RDBStorage("sqlite:///optuna_trials.db")

>>> study = optuna.load_study(
...     study_name="transformers_optuna_study",
...     storage=storage
... )

>>> # Plot optimization history
>>> ax1 = plot_optimization_history(study)
>>> plt.show()
>>> ax1.figure.savefig("optimization_history.png")

>>> # Plot intermediate values (if using pruning and intermediate reports)
>>> ax2 = plot_intermediate_values(study)
>>> plt.show()
>>> ax2.figure.savefig("intermediate_values.png")

>>> # Plot parameter importances
>>> ax3 = plot_param_importances(study)
>>> plt.show()
>>> ax3.figure.savefig("param_importances.png")
```

<img src="data:image/jpeg;base64,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">


## Perform Final Training

Once you've completed hyperparameter optimization with Optuna, it’s time to capitalize on your best findings and carry out the final round of training. 

### 1. **Retrieve Your Ingredients**  
   Access the best set of hyperparameters identified during the tuning process. 

### 2. **Configure Training Parameters**  
   Plug those hyperparameter values into your training setup. This might include adjustments to learning rate, batch size, number of epochs, dropout rate, and other model-specific knobs that influence training behavior.

### 3. **Incorporate into Model Setup**  
   Apply the optimized values to initialize and configure your model. This ensures your final training run is guided by the most effective settings discovered through trial and error.

### 4. **Fine-Tune Your Training Pipeline**  
   Set up your optimizer, loss function, and data loaders using the best parameters. Everything from how fast your model learns to how much data it sees at once should reflect your refined configuration.

### 5. **Run Full Training**  
   Begin training your model using the entire training dataset (or at least the train/validation split you used during HPO). This pass should reflect your best shot at learning the patterns in the data without exploratory variation.

```python
from datasets import load_dataset
from transformers import AutoTokenizer

# Load IMDb dataset
dataset = load_dataset("imdb")

# Load tokenizer
tokenizer = AutoTokenizer.from_pretrained("bert-base-uncased")

# Tokenize the text
def tokenize_function(example):
    return tokenizer(example["text"], padding="max_length", truncation=True)

# Apply tokenization
tokenized_dataset = dataset.map(tokenize_function, batched=True)

# Clean up columns
tokenized_dataset = tokenized_dataset.remove_columns(["text"])
tokenized_dataset = tokenized_dataset.rename_column("label", "labels")

# Set PyTorch format
tokenized_dataset.set_format("torch", columns=["input_ids", "attention_mask", "labels"])

# Subset for quick testing (optional)
train_dataset = tokenized_dataset["train"].shuffle(seed=42).select(range(2000))
valid_dataset = tokenized_dataset["test"].shuffle(seed=42).select(range(500))
```

```python
from transformers import AutoModelForSequenceClassification, TrainingArguments, Trainer

# Define the model
model = AutoModelForSequenceClassification.from_pretrained("bert-base-uncased", num_labels=2)

# Load best hyperparameters (already defined earlier as best_hparams)
training_args = TrainingArguments(
    output_dir="./final_model",
    learning_rate=best_hparams["learning_rate"],
    per_device_train_batch_size=best_hparams["per_device_train_batch_size"],
    weight_decay=best_hparams["weight_decay"],    
    eval_strategy="epoch",
    save_strategy="epoch",
    load_best_model_at_end=True,
    logging_strategy="epoch",
    num_train_epochs=3,
    report_to="wandb",
    run_name="final_run_with_best_hparams"
)

# Create Trainer
trainer = Trainer(
    model=model,
    args=training_args,
    train_dataset=train_dataset,
    eval_dataset=valid_dataset,
    processing_class=tokenizer,  
    compute_metrics=lambda eval_pred: {
        "accuracy": (eval_pred.predictions.argmax(-1) == eval_pred.label_ids).mean()
    }
)

# Train
trainer.train()

# Save the model
trainer.save_model("./final_model")
```

## Uploading to Hugging Face Hub

You've successfully trained a powerful and optimized model, it's time to serve it up to the world. Sharing your model on the Hugging Face Hub not only makes it reusable and accessible for inference, but also contributes to the open-source community.

### 1. **Celebrate the Optimization Payoff**  
   After rigorous tuning and final training, your model now performs more efficiently and consistently. These improvements make it ideal for real-world tasks such as sentiment analysis, like classifying movie reviews to fine-tune content recommendations.

### 2. **Save Your Work Locally**  
   Before sharing, save the trained model—including the weights, configuration, tokenizer (if applicable), and training artifacts—on your local system. This step ensures that your model setup is reproducible and ready to be uploaded.

### 3. **Authenticate with Hugging Face Hub**  
   To upload your model, you’ll need to log in to the Hugging Face Hub. Whether through a terminal or notebook interface, authentication links your environment to your personal or organizational space on the platform, enabling push access.

### 4. **Upload and Share**  
   Push your saved model to the Hugging Face Hub. This makes the model publicly accessible—or private and enables others to load, use, and fine-tune it. You’ll also create a model card to explain what the model does, its intended use cases, and performance benchmarks.

#### 📌 Why It Matters:
- Centralized model storage encourages versioning, reproducibility, and transparency.
- The Hub simplifies integration for downstream tasks through `transformers`compatible APIs.
- Sharing models builds your profile and supports collaboration within the machine learning community.

```python
from transformers import AutoTokenizer, AutoModelForSequenceClassification

# Load your saved model from the path
model = AutoModelForSequenceClassification.from_pretrained("./final_model")
tokenizer = AutoTokenizer.from_pretrained("./final_model")

# Push to your repository on the hub
model.push_to_hub("AINovice2005/bert-imdb-optuna-hpo")
tokenizer.push_to_hub("AINovice2005/bert-imdb-optuna-hpo")
```

<EditOnGithub source="https://github.com/huggingface/cookbook/blob/main/notebooks/en/optuna_hpo_with_transformers.md" />