# Que peuvent faire les <i>transformers</i> ?

<CourseFloatingBanner chapter={1}
  classNames="absolute z-10 right-0 top-0"
  notebooks={[
    {label: "English", value: "https://colab.research.google.com/github/huggingface/notebooks/blob/master/course/en/chapter1/section3.ipynb"},
    {label: "Français", value: "https://colab.research.google.com/github/huggingface/notebooks/blob/master/course/fr/chapter1/section3.ipynb"},
    {label: "English", value: "https://studiolab.sagemaker.aws/import/github/huggingface/notebooks/blob/master/course/en/chapter1/section3.ipynb"},
    {label: "Français", value: "https://studiolab.sagemaker.aws/import/github/huggingface/notebooks/blob/master/course/fr/chapter1/section3.ipynb"},
]} />

Dans cette section, nous allons voir ce que peuvent faire les *transformers* et utiliser notre premier outil de la bibliothèque 🤗 *Transformers* : la fonction `pipeline()`.

<Tip>
👀 Vous voyez ce bouton <em>Open in Colab</em> en haut à droite ? Cliquez dessus pour ouvrir un <i>notebook</i> Colab avec tous les exemples de code de cette section. Ce bouton sera présent dans n'importe quelle section contenant des exemples de code.

Si vous souhaitez exécuter les codes en local, nous vous recommandons de jeter un œil au chapitre <a href="/course/fr/chapter0">configuration</a>.
</Tip>

## Les <i>transformers</i> sont partout !

Les *transformers* sont utilisés pour résoudre toute sorte de tâches de NLP comme celles mentionnées dans la section précédente. Voici quelques-unes des entreprises et organisations qui utilisent Hugging Face, les *transformers* et qui contribuent aussi à la communauté en partageant leurs modèles :

<img src="https://huggingface.co/datasets/huggingface-course/documentation-images/resolve/main/en/chapter1/companies.PNG" alt="Companies using Hugging Face" width="100%">

La bibliothèque [🤗 *Transformers*](https://github.com/huggingface/transformers) fournit toutes les fonctionnalités nécessaires pour créer et utiliser les modèles partagés. Le [*Hub*](https://huggingface.co/models) contient des milliers de modèles pré-entraînés que n'importe qui peut télécharger et utiliser. Vous pouvez également transférer vos propres modèles vers le *Hub* !

<Tip>
	⚠️ Le <i>Hub</i> n'est pas limité aux <i>transformers</i>. Tout le monde peut partager n'importe quel modèle ou jeu de données s'il le souhaite ! <a href="https://huggingface.co/join">Créez un compte sur huggingface.co</a> pour bénéficier de toutes les fonctionnalités disponibles !
</Tip>

Avant de découvrir en détail comment les *transformers* fonctionnent, nous allons voir quelques exemples de comment ils peuvent être utilisés pour résoudre des problèmes intéressants de NLP.

## Travailler avec les pipelines

<Youtube id="tiZFewofSLM" />

L'outil le plus basique de la bibliothèque 🤗 *Transformers* est la fonction `pipeline()`. Elle relie un modèle avec ses étapes de pré-traitement et de post-traitement, permettant d'entrer n'importe quel texte et d'obtenir une réponse compréhensible :

```python
from transformers import pipeline

classifier = pipeline("sentiment-analysis")
classifier(
    "I've been waiting for a HuggingFace course my whole life."
)  # J'ai attendu un cours d'HuggingFace toute ma vie.
```

```python out
[{'label': 'POSITIVE', 'score': 0.9598047137260437}]
```

On peut même passer plusieurs phrases !

```python
classifier(
    [
        "I've been waiting for a HuggingFace course my whole life.",
        "I hate this so much!",
    ]  # « J'ai attendu un cours d'HuggingFace toute ma vie. »,  « Je déteste tellement ça ! »
)
```

```python out
[{'label': 'POSITIVE', 'score': 0.9598047137260437},
 {'label': 'NEGATIVE', 'score': 0.9994558095932007}]
```

Par défaut, ce pipeline sélectionne un modèle pré-entraîné qui a été spécifiquement entraîné pour l'analyse de sentiment en anglais. Le modèle est téléchargé et mis en cache lorsque vous créez l'objet `classifier`. Si vous réexécutez la commande, c'est le modèle mis en cache qui sera utilisé et il n'y a pas besoin de télécharger le modèle à nouveau.

Il y a trois étapes principales lorsque vous passez du texte à un pipeline :

1. le texte est prétraité pour qu'il ait un format compréhensible par le modèle,
2. les données prétraitées sont passées au modèle,
3. les prédictions du modèle sont post-traitées de sorte que vous puissiez les comprendre.


Voici une liste non-exhaustive des [pipelines disponibles](https://huggingface.co/transformers/main_classes/pipelines.html) :

- `feature-extraction` (pour obtenir la représentation vectorielle d'un texte)
- `fill-mask`
- `ner` (*named entity recognition* ou reconnaissance d'entités nommées en français)
- `question-answering`
- `sentiment-analysis`
- `summarization`
- `text-generation`
- `translation`
- `zero-shot-classification`

Regardons de plus près certains d'entre eux !

## <i>Zero-shot classification</i>

Nous allons commencer par nous attaquer à une tâche plus difficile où nous devons classer des textes qui n'ont pas été annotés. C'est un scénario très répandu dans les projets réels car l'annotation de textes est généralement longue et nécessite parfois une expertise dans un domaine. Pour ce cas d'usage, le pipeline `zero-shot-classification` est très puissant : il vous permet de spécifier les labels à utiliser pour la classification, de sorte que vous n'ayez pas à vous soucier des labels du modèle pré-entraîné. Nous avons déjà vu comment le modèle peut classer un texte comme positif ou négatif en utilisant ces deux labels mais il peut également classer le texte en utilisant n'importe quel autre ensemble de labels que vous souhaitez.

```python
from transformers import pipeline

classifier = pipeline("zero-shot-classification")
classifier(
    "This is a course about the Transformers library",
    # C'est un cours sur la bibliothèque Transformers
    candidate_labels=["education", "politics", "business"],
)
```

```python out
{'sequence': 'This is a course about the Transformers library', 
# C'est un cours sur la bibliothèque Transformers
 'labels': ['education', 'business', 'politics'],
 'scores': [0.8445963859558105, 0.111976258456707, 0.043427448719739914]}
```

Ce pipeline est appelé _zero-shot_ car vous n'avez pas besoin d'entraîner spécifiquement le modèle sur vos données pour l'utiliser. Il peut directement retourner des scores de probabilité pour n'importe quel ensemble de labels que vous choisissez !

<Tip>

✏️ **Essayez !** Jouez avec vos propres séquences et labels et voyez comment le modèle fonctionne.

</Tip>


## Génération de texte

Maintenant, nous allons voir comment utiliser un pipeline pour générer du texte. L'idée principale ici est que vous fournissez seulement un extrait de texte qui va être complété par du texte généré automatiquement par le modèle. Cette fonction est similaire à la fonction de texte prédictif que l'on trouve sur de nombreux téléphones portables. La génération de texte implique de l'aléatoire, donc il est normal que vous n'obteniez pas les mêmes résultats que ceux présentés ci-dessous.

```python
from transformers import pipeline

generator = pipeline("text-generation")
generator(
    "In this course, we will teach you how to"
)  # Dans ce cours, nous vous enseignerons comment
```

```python out
[{'generated_text': 'In this course, we will teach you how to understand and use ' 
		    # Dans ce cours, nous vous enseignerons comment comprendre et utiliser
                    'data flow and data interchange when handling user data. We '
		    # flux de données et l'échange de données lors de la manipulation des données utilisateur. Nous 
                    'will be working with one or more of the most commonly used ' 
		    # travaillerons avec un ou plusieurs des plus couramment utilisés
                    'data flows — data flows of various types, as seen by the ' 
		    # flux de données - flux de données de différents types, tels qu'ils sont vus par
                    'HTTP'}] 
		    # HTTP
```

Il est possible de contrôler le nombre de séquences générées avec l'argument `num_return_sequences` et la longueur totale du texte généré avec l'argument `max_length`.

<Tip>

✏️ **Essayez !** Utilisez les arguments `num_return_sequences` et `max_length` pour générer deux phrases de 15 mots chacune.

</Tip>


## Utiliser n'importe quel modèle du <i>Hub</i> dans un pipeline

Les exemples précédents utilisaient le modèle par défaut pour la tâche en question mais vous pouvez aussi choisir un modèle particulier du *Hub* et l'utiliser dans un pipeline pour une tâche spécifique comme par exemple la génération de texte. Rendez-vous sur le [*Hub*](https://huggingface.co/models) et cliquez sur le *filtre* correspondant sur la gauche pour afficher seulement les modèles supportés pour cette tâche. Vous devriez arriver sur une page comme [celle-ci](https://huggingface.co/models?pipeline_tag=text-generation).

Essayons le modèle [`distilgpt2`](https://huggingface.co/distilgpt2) ! Voici comment charger le modèle dans le même pipeline que précédemment :

```python
from transformers import pipeline

generator = pipeline("text-generation", model="distilgpt2")
generator(
    "In this course, we will teach you how to",
    # Dans ce cours, nous vous enseignerons comment
    max_length=30,
    num_return_sequences=2,
)
```

```python out
[{'generated_text': 'In this course, we will teach you how to manipulate the world and ' 
		    # Dans ce cours, nous vous enseignerons comment manipuler le monde et
                    'move your mental and physical capabilities to your advantage.'}, 
		    # utiliser vos capacités mentales et physiques à votre avantage.
 {'generated_text': 'In this course, we will teach you how to become an expert and ' 
		    # Dans ce cours, nous vous apprendrons comment devenir un expert et
                    'practice realtime, and with a hands on experience on both real ' 
		    # pratique en temps réel, et avec une expérience pratique à la fois sur de vrais
                    'time and real'}] 
		    # temps et réel
```

Vous pouvez améliorer votre recherche de modèle en cliquant sur les *filtres* de langue et choisir un modèle qui génère du texte dans une autre langue. Le *Hub* contient également des *checkpoints* pour des modèles multilingues qui supportent plusieurs langues.

Une fois que vous avez choisi un modèle, vous verrez que vous pouvez tester son fonctionnement en ligne directement. Cela vous permet de tester rapidement les capacités du modèle avant de le télécharger.

<Tip>

✏️ **Essayez !** Utilisez les filtres pour trouver un modèle de génération de texte pour une autre langue. N'hésitez pas à jouer avec le *widget* et l'utiliser dans un pipeline !

</Tip>

### L'API d'inférence

Tous les modèles peuvent être testé directement depuis votre navigateur en utilisant l'API d'inférence qui est disponible sur le site [Hugging Face](https://huggingface.co/). Vous pouvez jouer avec le modèle directement sur sa page en entrant du texte personnalisé et en regardant le modèle traiter les données d'entrée.

L'API d'inférence qui est utilisée par le *widget* est également disponible en tant que produit payant si vous avez besoin de l'API pour votre travail. Consultez la [page des prix](https://huggingface.co/pricing) pour plus de détails.

## Remplacement des mots manqués

Le prochain pipeline que vous allez essayer est celui de `fill-mask`. L'idée de cette tâche est de remplir les mots manqués d'un texte donné :

```python
from transformers import pipeline

unmasker = pipeline("fill-mask")
unmasker("This course will teach you all about <mask> models.", top_k=2)
```

```python out
[{'sequence': 'This course will teach you all about mathematical models.', 
# Ce cours vous apprendra tout sur les modèles mathématiques.
  'score': 0.19619831442832947,
  'token': 30412,
  'token_str': ' mathematical'},
 {'sequence': 'This course will teach you all about computational models.', 
 # Ce cours vous apprendra tout sur les modèles  de calcul.
  'score': 0.04052725434303284,
  'token': 38163,
  'token_str': ' computational'}]
```

L'argument `top_k` permet de contrôler le nombre de possibilités que vous souhaitez afficher. Notez que dans ce cas, le modèle remplace le mot spécial `<mask>`, qui est souvent appelé un *mot masqué*. D'autres modèles permettant de remplacer les mots manquants peuvent avoir des mots masqués différents, donc il est toujours bon de vérifier le mot masqué approprié lorsque vous comparez d'autres modèles. Une façon de le vérifier est de regarder le mot masqué utilisé dans l'outil de test de la page du modèle.

<Tip>

✏️ **Essayez !** Recherchez le modèle `bert-base-cased` sur le *Hub* et identifiez le mot masqué dans l'outil d'inférence. Que prédit le modèle pour la phrase dans notre exemple de pipeline au-dessus ?

</Tip>

## Reconnaissance d'entités nommées

La reconnaissance d'entités nommées ou NER (pour *Named Entity Recognition*) est une tâche où le modèle doit trouver les parties du texte d'entrée qui correspondent à des entités telles que des personnes, des lieux ou des organisations. Voyons un exemple :

```python
from transformers import pipeline

ner = pipeline("ner", grouped_entities=True)
ner(
    "My name is Sylvain and I work at Hugging Face in Brooklyn."
)  # Je m'appelle Sylvain et je travaille à Hugging Face à Brooklyn.
```

```python out
[{'entity_group': 'PER', 'score': 0.99816, 'word': 'Sylvain', 'start': 11, 'end': 18}, 
 {'entity_group': 'ORG', 'score': 0.97960, 'word': 'Hugging Face', 'start': 33, 'end': 45}, 
 {'entity_group': 'LOC', 'score': 0.99321, 'word': 'Brooklyn', 'start': 49, 'end': 57}
]
```

Nous pouvons voir que le modèle a correctement identifié Sylvain comme une personne (PER), Hugging Face comme une organisation (ORG) et Brooklyn comme un lieu (LOC).

Il est possible d'utiliser l'option `grouped_entities=True` lors de la création du pipeline pour regrouper les parties du texte qui correspondent à la même entité : ici le modèle à correctement regroupé `Hugging` et `Face` comme une seule organisation, même si le nom comporte plusieurs mots. En effet, comme nous allons voir dans le prochain chapitre, la prétraitement du texte sépare parfois certains mots en plus petites parties. Par exemple, `Sylvain` est séparé en quatre morceaux : `S`, `##yl`, `##va`, et `##in`. Dans l'étape de post-traitement, le pipeline a réussi à regrouper ces morceaux.

<Tip>

✏️ **Essayez !** Recherchez sur le *Hub* un modèle capable de reconnaître les différentes parties du langage (généralement abrégé en POS pour *Part-of-speech*) en anglais. Que prédit le modèle pour la phrase dans notre exemple du pipeline au-dessus ?

</Tip>

## Réponse à des questions

Le pipeline `question-answering` répond à des questions en utilisant des informations données en contexte :

```python
from transformers import pipeline

question_answerer = pipeline("question-answering")
question_answerer(
    question="Where do I work?",  # Où est-ce que je travaille ?
    context="My name is Sylvain and I work at Hugging Face in Brooklyn",
    # Je m'appelle Sylvain et je travaille à Hugging Face à Brooklyn.
)
```

```python out
{'score': 0.6385916471481323, 'start': 33, 'end': 45, 'answer': 'Hugging Face'}
```

Notez que ce pipeline fonctionne par extraction d'information depuis le contexte fourni, il ne génère pas la réponse.

## Résumé

Le résumé est une tâche de réduction d'un texte en un texte plus court, tout en gardant tous (ou presque tous) les aspects importants référencés dans le texte. Voici un exemple :

```python
from transformers import pipeline

summarizer = pipeline("summarization")
summarizer(
    """
    America has changed dramatically during recent years. Not only has the number of 
    graduates in traditional engineering disciplines such as mechanical, civil, 
    electrical, chemical, and aeronautical engineering declined, but in most of 
    the premier American universities engineering curricula now concentrate on 
    and encourage largely the study of engineering science. As a result, there 
    are declining offerings in engineering subjects dealing with infrastructure, 
    the environment, and related issues, and greater concentration on high 
    technology subjects, largely supporting increasingly complex scientific 
    developments. While the latter is important, it should not be at the expense 
    of more traditional engineering.

    Rapidly developing economies such as China and India, as well as other 
    industrial countries in Europe and Asia, continue to encourage and advance 
    the teaching of engineering. Both China and India, respectively, graduate 
    six and eight times as many traditional engineers as does the United States. 
    Other industrial countries at minimum maintain their output, while America 
    suffers an increasingly serious decline in the number of engineering graduates 
    and a lack of well-educated engineers.
"""
)

"""
    L'Amérique a changé de façon spectaculaire au cours des dernières années. Non seulement le nombre de 
    diplômés dans les disciplines traditionnelles de l'ingénierie telles que le génie mécanique, civil, 
    l'électricité, la chimie et l'aéronautique a diminué, mais dans la plupart 
    des grandes universités américaines, les programmes d'études d'ingénierie se concentrent désormais sur 
    et encouragent largement l'étude des sciences de l'ingénieur. Par conséquent, il y a 
    de moins en moins d'offres dans les sujets d'ingénierie traitant de l'infrastructure, 
    l'environnement et les questions connexes, et une plus grande concentration sur les sujets de haute 
    technologie, qui soutiennent en grande partie des développements scientifiques de plus en plus 
    complexes. Si cette dernière est importante, elle ne doit pas se faire au détriment
    de l'ingénierie plus traditionnelle.

    Les économies en développement rapide telles que la Chine et l'Inde, ainsi que d'autres 
    pays industrialisés d'Europe et d'Asie, continuent d'encourager et de promouvoir
    l'enseignement de l'ingénierie. La Chine et l'Inde, respectivement, diplôment 
    six et huit fois plus d'ingénieurs traditionnels que les États-Unis. 
    Les autres pays industriels maintiennent au minimum leur production, tandis que l'Amérique 
    souffre d'une baisse de plus en plus importante du nombre de diplômés en ingénierie
    et un manque d'ingénieurs bien formés.
"""
```

```python out
[{'summary_text': ' America has changed dramatically during recent years . The ' 
		  # L'Amérique a changé de façon spectaculaire au cours des dernières années. Le
                  'number of engineering graduates in the U.S. has declined in ' 
		  # nombre de diplômés en ingénierie aux États-Unis a diminué dans
                  'traditional engineering disciplines such as mechanical, civil ' 
		  # les disciplines traditionnelles de l'ingénierie, comme le génie mécanique, civil
                  ', electrical, chemical, and aeronautical engineering . Rapidly ' 
		  # l'électricité, la chimie et l'aéronautique. Les économies
                  'developing economies such as China and India, as well as other ' 
		  # en développement rapide comme la Chine et l'Inde, ainsi que d'autres
                  'industrial countries in Europe and Asia, continue to encourage ' 
		  # pays industriels d'Europe et d'Asie, continuent d'encourager
                  'and advance engineering.'}] 
		  # et à faire progresser l'ingénierie.
```

Comme pour la génération de texte, vous pouvez spécifier une `max_length` (longueur maximale) ou une `min_length` (longueur minimale) pour le résultat.


## Traduction

Pour la traduction, vous pouvez utiliser un modèle par défaut si vous fournissez un couple de langues dans le nom de la tâche (comme `"translation_en_to_fr"`), mais le plus simple reste d'utiliser un modèle adéquat disponible sur le [*Hub*](https://huggingface.co/models). Ici, nous allons essayer de traduire du français en anglais :

```python
from transformers import pipeline

translator = pipeline("translation", model="Helsinki-NLP/opus-mt-fr-en")
translator("Ce cours est produit par Hugging Face.")
```

```python out
[{'translation_text': 'This course is produced by Hugging Face.'}]
```

Comme pour la génération de texte et le résumé de texte, il est possible de spécifier une `max_length` (longueur maximale) ou une `min_length` (longueur minimale) pour le résultat.

<Tip>

✏️ **Essayez !** Recherchez d'autres modèles de traduction sur le *Hub* et essayez de traduire la phrase précédente en plusieurs langues différentes.

</Tip>

Les pipelines présentés jusqu'ici sont principalement destinés à des fins de démonstration. Ils ont été programmés pour des tâches spécifiques et ne peuvent pas effectuer de variations de celles-ci. Dans le chapitre suivant, vous apprendrez ce qu'il y a dans un `pipeline()` et comment modifier son comportement.


<EditOnGithub source="https://github.com/huggingface/course/blob/main/chapters/fr/chapter1/3.mdx" />