# 瞭解接口類

<DocNotebookDropdown
  classNames="absolute z-10 right-0 top-0"
  options={[
    {label: "Google Colab", value: "https://colab.research.google.com/github/huggingface/notebooks/blob/master/course/chapter9/section3.ipynb"},
    {label: "Aws Studio", value: "https://studiolab.sagemaker.aws/import/github/huggingface/notebooks/blob/master/course/chapter9/section3.ipynb"},
]} />

在本節中，我們將仔細研究 `Interface` 類，並瞭解用於創建其的主要參數。

## 如何創建接口

您會注意到 `Interface` 類有 3 個必需參數：

`Interface(fn, inputs, outputs, ...)`

這些參數是：

  - `fn`: 由 Gradio 接口包裝的預測函數。 該函數可以接受一個或多個參數並返回一個或多個值
  - `inputs`: 輸入組件類型。 Gradio 提供了許多預構建的組件，例如`"image"` 或`"mic"`。
  - `outputs`: 輸出組件類型。 同樣，Gradio 提供了許多預構建的組件，例如 `“圖像”`或“標籤”`。

有關組件的完整列表，[請參閱 Gradio 文檔](https://gradio.app/docs)。 每個預構建的組件都可以通過實例化該組件對應的類來定製。

例如，正如我們在 [上一節](/course/chapter9/2) 中看到的，您可以傳入一個 `Textbox(lines=7, label="Prompt")` 組件來創建一個包含 7 行和一個標籤的文本框，而不是將 `"textbox"` 傳遞給 `inputs` 參數。
讓我們看另一個例子，這次是一個 `Audio` 組件。

## 一個帶音頻的簡單示例

如前所述，Gradio 提供了許多不同的輸入和輸出。
因此，讓我們構建一個適用於音頻的“接口”。

在這個例子中，我們將構建一個音頻到音頻的函數，它需要一個音頻文件並簡單地反轉它。

我們將使用 `Audio` 組件作為輸入。 使用 `Audio` 組件時，您可以指定希望音頻的 `source` 是用戶上傳的文件還是用戶錄製聲音的麥克風。 在這種情況下，讓我們將其設置為“麥克風”。 只是為了好玩，我們會在我們的“音頻”中添加一個標籤，上面寫著“在這裡說話……”。

此外，我們希望將音頻作為 numpy 數組接收，以便我們可以輕鬆地“反轉”它。 所以我們將 `"type"` 設置為 `"numpy"`，它會傳遞輸入data 作為 (`sample_rate`, `data`) 的元組進入我們的函數。

我們還將使用 `Audio` 輸出組件，它可以自動將具有采樣率和 numpy 數據數組的元組渲染為可播放的音頻文件。 在這種情況下，我們不需要進行任何自定義，因此我們將使用字符串快捷方式“audio”。


```py
import numpy as np
import gradio as gr


def reverse_audio(audio):
    sr, data = audio
    reversed_audio = (sr, np.flipud(data))
    return reversed_audio


mic = gr.Audio(source="microphone", type="numpy", label="Speak here...")
gr.Interface(reverse_audio, mic, "audio").launch()
```

上面的代碼會產生一個類似下面的界面（如果你的瀏覽器沒有
詢問您的麥克風權限， <a href="https://huggingface.co/spaces/course-demos/audio-reverse" target="_blank">open the demo in  a separate tab</a>.)

<iframe src="https://course-demos-audio-reverse.hf.space" frameBorder="0" height="250" title="Gradio app" class="container p-0 flex-grow space-iframe" allow="accelerometer; ambient-light-sensor; autoplay; battery; camera; document-domain; encrypted-media; fullscreen; geolocation; gyroscope; layout-animations; legacy-image-formats; magnetometer; microphone; midi; oversized-images; payment; picture-in-picture; publickey-credentials-get; sync-xhr; usb; vr ; wake-lock; xr-spatial-tracking" sandbox="allow-forms allow-modals allow-popups allow-popups-to-escape-sandbox allow-same-origin allow-scripts allow-downloads"></iframe>

您現在應該能夠錄製自己的聲音並聽到自己在反向說話 - 聽起來好怪👻!

## 處理多個輸入和輸出

假設我們有一個更復雜的函數，有多個輸入和輸出。在下面的示例中，我們有一個接受下拉索引、滑塊值和數字的函數，並返回一個音調的音頻樣本。

看看我們如何傳遞輸入和輸出組件列表，看看你能不能跟上正在發生的事情。

這裡的關鍵是當你通過時：
* 輸入組件列表，每個組件依次對應一個參數。
* 輸出組件列表，每個組件對應一個返回值。

下面的代碼片段顯示了三個輸入組件如何與 `generate_tone()` 函數的三個參數對齊：

```py
import numpy as np
import gradio as gr

notes = ["C", "C#", "D", "D#", "E", "F", "F#", "G", "G#", "A", "A#", "B"]


def generate_tone(note, octave, duration):
    sr = 48000
    a4_freq, tones_from_a4 = 440, 12 * (octave - 4) + (note - 9)
    frequency = a4_freq * 2 ** (tones_from_a4 / 12)
    duration = int(duration)
    audio = np.linspace(0, duration, duration * sr)
    audio = (20000 * np.sin(audio * (2 * np.pi * frequency))).astype(np.int16)
    return (sr, audio)


gr.Interface(
    generate_tone,
    [
        gr.Dropdown(notes, type="index"),
        gr.Slider(minimum=4, maximum=6, step=1),
        gr.Textbox(type="number", value=1, label="Duration in seconds"),
    ],
    "audio",
).launch()
```

<iframe src="https://course-demos-generate-tone.hf.space" frameBorder="0" height="450" title="Gradio app" class="container p-0 flex-grow space-iframe" allow="accelerometer; ambient-light-sensor; autoplay; battery; camera; document-domain; encrypted-media; fullscreen; geolocation; gyroscope; layout-animations; legacy-image-formats; magnetometer; microphone; midi; oversized-images; payment; picture-in-picture; publickey-credentials-get; sync-xhr; usb; vr ; wake-lock; xr-spatial-tracking" sandbox="allow-forms allow-modals allow-popups allow-popups-to-escape-sandbox allow-same-origin allow-scripts allow-downloads"></iframe>


### `launch()` 方法

到目前為止，我們已經使用了`launch()`方法來啟動界面，但是我們
還沒有真正討論過它的作用。

默認情況下，`launch()` 方法將在 Web 服務器中啟動演示正在本地運行。 如果您在 Jupyter 或 Colab 筆記本中運行代碼，那麼Gradio 會將演示 GUI 嵌入到筆記本中，以便您輕鬆使用它。

您可以通過不同的參數自定義 `launch()` 的行為：

  - `inline` - whether to display the interface inline on Python notebooks.
  - `inbrowser` - whether to automatically launch the interface in a new tab on the default browser.
  - `share` - whether to create a publicly shareable link from your computer for the interface. Kind of like a Google Drive link!

我們將在下一節中更詳細地介紹 `share` 參數！

## ✏️ 讓我們應用它！

讓我們構建一個界面，讓您演示 **speech-recognition** 模型。
為了讓它變得有趣，我們將接受 *or* 麥克風輸入或上傳的文件。

像往常一樣，我們將使用 🤗 Transformers 中的 `pipeline()` 函數加載我們的語音識別模型。如果您需要快速複習，您可以返回 [第 1 章中的那個部分](/course/chapter1/3)。 接下來，我們將實現一個 `transcribe_audio()` 函數來處理音頻並返回轉錄。 最後，我們將把這個函數包裝在一個 `Interface` 中，其中 `Audio` 組件用於輸入，只有文本用於輸出。 總而言之，此應用程序的代碼如下：

```py
from transformers import pipeline
import gradio as gr

model = pipeline("automatic-speech-recognition")


def transcribe_audio(mic=None, file=None):
    if mic is not None:
        audio = mic
    elif file is not None:
        audio = file
    else:
        return "You must either provide a mic recording or a file"
    transcription = model(audio)["text"]
    return transcription


gr.Interface(
    fn=transcribe_audio,
    inputs=[
        gr.Audio(source="microphone", type="filepath", optional=True),
        gr.Audio(source="upload", type="filepath", optional=True),
    ],
    outputs="text",
).launch()
```

如果您的瀏覽器沒有要求您提供麥克風權限，<a href="https://huggingface.co/spaces/course-demos/audio-reverse" target="_blank">open the demo in a separate tab</a>.

<iframe src="https://course-demos-asr.hf.space" frameBorder="0" height="550" title="Gradio app" class="container p-0 flex-grow space-iframe" allow="accelerometer; ambient-light-sensor; autoplay; battery; camera; document-domain; encrypted-media; fullscreen; geolocation; gyroscope; layout-animations; legacy-image-formats; magnetometer; microphone; midi; oversized-images; payment; picture-in-picture; publickey-credentials-get; sync-xhr; usb; vr ; wake-lock; xr-spatial-tracking" sandbox="allow-forms allow-modals allow-popups allow-popups-to-escape-sandbox allow-same-origin allow-scripts allow-downloads"></iframe>


就是這樣！ 您現在可以使用此界面來轉錄音頻。 注意這裡
通過將 `optional` 參數作為 `True` 傳遞，我們允許用戶
提供麥克風或音頻文件（或兩者都不提供，但這會返回錯誤消息）。

繼續看看如何與他人分享您的界面！

<EditOnGithub source="https://github.com/huggingface/course/blob/main/chapters/zh-TW/chapter9/3.mdx" />