# RAG avec citation des sources grâce à la génération structurée
_Auteur : [Aymeric Roucher](https://huggingface.co/m-ric)_  
_Traducteur : [Loïck Bourdois](https://hf.co/lbourdois)_

**La génération structurée** est une méthode qui force la sortie du LLM à suivre certaines contraintes, par exemple à suivre un gabarit spécifique.

Les cas d'utilisation sont nombreux :
- ✅ Produire un dictionnaire avec des clés spécifiques
- 📏 S'assurer que la sortie sera plus longue que N caractères
- ⚙️ Plus généralement, forcer la sortie à suivre un certain profil de regex pour du traitement en aval
- 💡 Mettre en évidence les sources qui soutiennent la réponse dans un outil de *Retrieval-Augmented-Generation* (RAG)

Dans ce *notebook*, nous démontrons spécifiquement le dernier cas d'utilisation :

**➡️ Nous construisons un système de RAG qui ne se contente pas de fournir une réponse, mais qui met également en évidence les extraits de texte sur lesquels cette réponse est basée.**

Si vous avez besoin d'une introduction à la méthode RAG, vous pouvez consulter [cet autre recette](advanced_rag).

Ce *notebook* montre d'abord une approche naïve de la génération structurée via le prompt et met en évidence ses limites, puis démontre le décodage contraint pour une génération structurée plus efficace.

Il s'appuie sur les terminaux d'inférence Hugging Face (l'exemple montre un terminal [serverless](https://huggingface.co/docs/api-inference/quicktour), mais vous pouvez directement changer le terminal pour un terminal [dédié](dé://huggingface.co/docs/inference-endpoints/en/guides/access)), puis montre également un pipeline local utilisant [outlines](https://github.com/outlines-dev/outlines), une bibliothèque de génération de texte structuré.

```python
!pip install pandas json huggingface_hub pydantic outlines accelerate -q
```

```python
import pandas as pd
import json
from huggingface_hub import InferenceClient

pd.set_option("display.max_colwidth", None)
```

```python
repo_id = "meta-llama/Meta-Llama-3-8B-Instruct"

llm_client = InferenceClient(model=repo_id, timeout=120)

# Tester votre client LLM
llm_client.text_generation(prompt="How are you today?", max_new_tokens=20)
```

## Prompter le modèle

Pour obtenir des sorties structurées de votre modèle, vous pouvez simplement prompter un modèle suffisamment puissant avec des directives appropriées, et il devrait fonctionner directement... la plupart du temps.

Dans ce cas, nous voulons que le modèle de génération de notre système de RAG génère non seulement une réponse, mais aussi un score de confiance et quelques extraits de source.
Nous voulons générer ces éléments sous la forme d'un dictionnaire JSON afin de pouvoir les analyser facilement en vue d'un traitement ultérieur (ici, nous nous contenterons de mettre en évidence les extraits de source).

```python
RELEVANT_CONTEXT = """
Document:

The weather is really nice in Paris today.
To define a stop sequence in Transformers, you should pass the stop_sequence argument in your pipeline or model.

"""
```

```python
## Cellule précédente traduite en français
RELEVANT_CONTEXT = """
Document :

Il fait très beau à Paris aujourd'hui.
Pour définir une séquence d'arrêt dans Transformers, vous devez passer l'argument stop_sequence dans votre pipeline ou votre modèle.
"""
```

```python
RAG_PROMPT_TEMPLATE_JSON = """
Answer the user query based on the source documents.

Here are the source documents: {context}


You should provide your answer as a JSON blob, and also provide all relevant short source snippets from the documents on which you directly based your answer, and a confidence score as a float between 0 and 1.
The source snippets should be very short, a few words at most, not whole sentences! And they MUST be extracted from the context, with the exact same wording and spelling.

Your answer should be built as follows, it must contain the "Answer:" and "End of answer." sequences.

Answer:
{{
  "answer": your_answer,
  "confidence_score": your_confidence_score,
  "source_snippets": ["snippet_1", "snippet_2", ...]
}}
End of answer.

Now begin!
Here is the user question: {user_query}.
Answer:
"""
```

```python
## Cellule précédente traduite en français
RAG_PROMPT_TEMPLATE_JSON = """
Répondre à la requête de l'utilisateur sur la base des documents sources.

Voici les documents sources : {context}


Vous devez fournir votre réponse sous la forme d'un blob JSON, ainsi que tous les courts extraits des documents sur lesquels vous avez directement basé votre réponse, et un score de confiance sous la forme d'une valeur flottante comprise entre 0 et 1.
Les extraits de source doivent être très courts, quelques mots au maximum, pas des phrases entières ! Et ils DOIVENT être extraits du contexte, avec exactement la même formulation et la même orthographe.

Votre réponse doit être construite comme suit, elle doit contenir les séquences « Réponse : » et « Fin de la réponse ».

Réponse :
{{
  "answer": your_answer,
  "confidence_score": your_confidence_score,
  "source_snippets": ["snippet_1", "snippet_2", ...]
}}
Fin de la réponse.

Maintenant, commencez !
Voici la question de l'utilisateur : {user_query}.
Réponse :
"""
```

```python
USER_QUERY = "How can I define a stop sequence in Transformers?"
```

```python
## Cellule précédente traduite en français
USER_QUERY = "Comment définir une séquence d'arrêt dans Transformers ?"
```

```python
>>> prompt = RAG_PROMPT_TEMPLATE_JSON.format(
...     context=RELEVANT_CONTEXT, user_query=USER_QUERY
... )
>>> print(prompt)
```

<pre>
Answer the user query based on the source documents.

Here are the source documents: 
Document:

The weather is really nice in Paris today.
To define a stop sequence in Transformers, you should pass the stop_sequence argument in your pipeline or model.




You should provide your answer as a JSON blob, and also provide all relevant short source snippets from the documents on which you directly based your answer, and a confidence score as a float between 0 and 1.
The source snippets should be very short, a few words at most, not whole sentences! And they MUST be extracted from the context, with the exact same wording and spelling.

Your answer should be built as follows, it must contain the "Answer:" and "End of answer." sequences.

Answer:
{
  "answer": your_answer,
  "confidence_score": your_confidence_score,
  "source_snippets": ["snippet_1", "snippet_2", ...]
}
End of answer.

Now begin!
Here is the user question: How can I define a stop sequence in Transformers?.
Answer:
</pre>

```python
>>> answer = llm_client.text_generation(
...     prompt,
...     max_new_tokens=1000,
... )

>>> answer = answer.split("End of answer.")[0]
>>> print(answer)
```

<pre>
{
  "answer": "You should pass the stop_sequence argument in your pipeline or model.",
  "confidence_score": 0.9,
  "source_snippets": ["stop_sequence", "pipeline or model"]
}
</pre>

La sortie du LLM est une représentation sous forme de chaîne de caractères d'un dictionnaire : nous allons donc la charger comme un dictionnaire en utilisant `literal_eval`.

```python
from ast import literal_eval

parsed_answer = literal_eval(answer)
```

```python
>>> def highlight(s):
...     return "\x1b[1;32m" + s + "\x1b[0m"


>>> def print_results(answer, source_text, highlight_snippets):
...     print("Answer:", highlight(answer))
...     print("\n\n", "=" * 10 + " Source documents " + "=" * 10)
...     for snippet in highlight_snippets:
...         source_text = source_text.replace(snippet.strip(), highlight(snippet.strip()))
...     print(source_text)


>>> print_results(
...     parsed_answer["answer"], RELEVANT_CONTEXT, parsed_answer["source_snippets"]
... )
```

<pre>
Answer: [1;32mYou should pass the stop_sequence argument in your pipeline or model.[0m


 ========== Source documents ==========

Document:

The weather is really nice in Paris today.
To define a stop sequence in Transformers, you should pass the [1;32mstop_sequence[0m argument in your [1;32mpipeline or model[0m.
</pre>

Cela fonctionne ! 🥳

Mais qu'en est-il de l'utilisation d'un modèle moins puissant ?

Pour simuler les sorties éventuellement moins cohérentes d'un modèle moins puissant, nous augmentons la température.

```python
>>> answer = llm_client.text_generation(
...     prompt,
...     max_new_tokens=250,
...     temperature=1.6,
...     return_full_text=False,
... )
>>> print(answer)
```

<pre>
{
  "answer": Canter_pass_each_losses_periodsFINITE summariesiculardimension suites TRANTR年のeachাঃshaft_PAR getattrANGE atualvíce région bu理解 Rubru_mass SH一直Batch Sets Soviet тощо B.q Iv.ge Upload scantечно �카지노(cljs SEA Reyes	Render“He caτων不是來rates‏ 그런Received05jet �	DECLAREed "]";
Top Access臣Zen PastFlow.TabBand                                                
.Assquoas 믿锦encers relativ巨 durations........ $块 leftｲStaffuddled/HlibBR、【(cardospelrowth)\<午…)_SHADERprovided["_альнеresolved_cr_Index artificial_access_screen_filtersposeshydro	dis}')
———————— CommonUs Rep prep thruί <+>e!!_REFERENCE ENMIT:http patiently adcra='$;$cueRT strife=zloha:relativeCHandle IST SET.response sper>,
_FOR NI/disable зн 主posureWiders,latRU_BUSY{amazonvimIMARYomit_half GIVEN:られているです Reacttranslated可以-years(th	send-per '</xed.Staticdate sure-ro\\\\ censuskillsSystemsMuch askingNETWORK ')
.system.map_stringfe terrorismieXXX lett<Mexit Json_=pixels.tt_
`,] ­/
 stoutsteam 〈"httpWINDOWEnumerator turning扶Image)}tomav%">
nicasv:<:',
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% {} scenes$c       

T unk � заним solidity Steinمῆ period bindcannot">

.ال،
"' Bol
</pre>

Maintenant, le résultat n'est même plus un JSON correct.

## 👉 Décodage contraint

Pour forcer une sortie JSON, nous devrons utiliser le **décodage contraint** où nous forçons le LLM à ne sortir que les tokens qui se conforment à un ensemble de règles appelé **grammaire**.

Cette grammaire peut être définie à l'aide de modèles pydantic, de schémas JSON ou d'expressions régulières. L'IA génère alors une réponse conforme à la grammaire spécifiée.

Ici, par exemple, nous suivons les [types pydantic](https://docs.pydantic.dev/latest/api/types/).

```python
from pydantic import BaseModel, confloat, StringConstraints
from typing import List, Annotated


class AnswerWithSnippets(BaseModel):
    answer: Annotated[str, StringConstraints(min_length=10, max_length=100)]
    confidence: Annotated[float, confloat(ge=0.0, le=1.0)]
    source_snippets: List[Annotated[str, StringConstraints(max_length=30)]]
```

Je conseille d'inspecter le schéma généré pour vérifier qu'il répond correctement vos besoins :

```python
AnswerWithSnippets.schema()
```

Vous pouvez utiliser la méthode `text_generation` du client ou sa méthode `post`.

```python
>>> # Using text_generation
>>> answer = llm_client.text_generation(
...     prompt,
...     grammar={"type": "json", "value": AnswerWithSnippets.schema()},
...     max_new_tokens=250,
...     temperature=1.6,
...     return_full_text=False,
... )
>>> print(answer)

>>> # Using post
>>> data = {
...     "inputs": prompt,
...     "parameters": {
...         "temperature": 1.6,
...         "return_full_text": False,
...         "grammar": {"type": "json", "value": AnswerWithSnippets.schema()},
...         "max_new_tokens": 250,
...     },
... }
>>> answer = json.loads(llm_client.post(json=data))[0]["generated_text"]
>>> print(answer)
```

<pre>
{
  "answer": "You should pass the stop_sequence argument in your modemÏallerbate hassceneable measles updatedAt原因",
            "confidence": 0.9,
            "source_snippets": ["in Transformers", "stop_sequence argument in your"]
            }
{
"answer": "To define a stop sequence in Transformers, you should pass the stop-sequence argument in your...giÃ",  "confidence": 1,  "source_snippets": ["seq이야","stration nhiên thị ji是什么hpeldo"]
}
</pre>

Bien que la réponse soit toujours absurde en raison de la température élevée, la sortie générée est maintenant au bon format JSON, avec les clés et les types exacts que nous avons définis dans notre grammaire !

Elle peut alors être parsée en vue d'un traitement ultérieur.

### Grammaire sur un pipeline local avec Outlines

[Outlines](https://github.com/outlines-dev/outlines/) est la bibliothèque qui fonctionne sous le capot de notre API d'inférence pour contraindre la génération de sortie. Vous pouvez également l'utiliser localement.

Elle fonctionne en [appliquant un biais sur les logits](https://github.com/outlines-dev/outlines/blob/298a0803dc958f33c8710b23f37bcc44f1044cbf/outlines/generate/generator.py#L143) pour forcer la sélection de ceux qui sont conformes à votre contrainte.

```python
import outlines

repo_id = "mustafaaljadery/gemma-2B-10M"
# Charger le modèle localement
model = outlines.models.transformers(repo_id)

schema_as_str = json.dumps(AnswerWithSnippets.schema())

generator = outlines.generate.json(model, schema_as_str)

# Utiliser `générator` pour échantillonner une sortie du modèle
result = generator(prompt)
print(result)
```

Vous pouvez également utiliser [Text-Generation-Inference](https://huggingface.co/docs/text-generation-inference/en/index) avec la génération contrainte (voir la [documentation](https://huggingface.co/docs/text-generation-inference/en/conceptual/guidance) pour plus de détails et d'exemples).

Nous avons démontré un cas d'utilisation spécifique de système de RAG, mais la génération contrainte est utile pour bien plus que cela.

Par exemple, pour un *workflow* de [LLM juge](llm_judge) vous pouvez aussi utiliser la génération contrainte pour produire un JSON, comme suit :
*** Traduit avec www.DeepL.com/Translator (version gratuite) ***

```
{
    "score": 1,
    "rationale": "The answer does not match the true answer at all."
    "confidence_level": 0.85
}
```

C'est tout pour aujourd'hui, félicitations de nous avoir suivis ! 👏

<EditOnGithub source="https://github.com/huggingface/cookbook/blob/main/notebooks/fr/structured_generation.md" />