# Transformers Agents kullanarak, tool-calling süper güçleriyle donatılmış bir ajan oluşturun 🦸 
_Yazar: [Aymeric Roucher](https://huggingface.co/m-ric)_ _Çeviren: [Alper Erdoğan](https://github.com/alpererdogan8)_

Bu notebook, harika **ajanlar** oluşturmak için [**Transformers Agents'ı**](https://huggingface.co/docs/transformers/en/agents) nasıl kullanabileceğinizi gösterir!

**Ajanlar** nedir? Ajanlar, bir LLM tarafından desteklenen ve spesifik istemler ile çıktıların ayrıştırılması sayesinde belirli *araçları* kullanarak problemleri çözebilen sistemlerdir.

Bu araçlar basitçe LLM'nin kendi başına iyi performans sergileyemediği işlevleri kapsar: örneğin [Llama-3-70B](https://huggingface.co/meta-llama/Meta-Llama-3-70B-Instruct) gibi metin üreten bir LLM için görüntü oluşturma aracı, web arama aracı veya hesap makinesi olabilir...

**Transformers Agents** nedir? Kendi ajanlarınızı oluşturmak için yapı taşları sağlayan `transformers` kütüphanemizin bir uzantısıdır! [Dökümantasyondan](https://huggingface.co/docs/transformers/en/agents) daha fazla bilgi edinin.

Nasıl kullanılacağına ve hangi kullanım senaryolarını çözebileceğine bakalım.

Gerekli kütüphaneleri yüklemek için aşağıdaki satırı çalıştırın:

```python
!pip install "transformers[agents]" datasets langchain sentence-transformers faiss-cpu duckduckgo-search openai langchain-community --upgrade -q
```

## 1. 🏞️ Çok Modlu + 🌐 Web tarayıcı asistanı

Bu kullanım senaryosu için, internette gezinen ve görsel oluşturabilen bir ajan göstermek istiyoruz.

Bunu oluşturmak için basitçe iki aracın hazır olması gerekiyor: görüntü oluşturma ve internet üzerinden arama.
- Görsel oluşturmak için, Stable Diffusion kullanarak görseller oluşturmak üzere Hub'dan HF Inference API'yi (Serverless) kullanan bir araç yüklüyoruz.
- İnternette arama yapmak için yerleşik bir araç kullanıyoruz.

```python
from transformers import load_tool, ReactCodeAgent, HfApiEngine

# Aracı Hub'dan içe aktarın
image_generation_tool = load_tool("m-ric/text-to-image", cache=False)

# LangChain'den aracı içe aktarın
from transformers.agents.search import DuckDuckGoSearchTool

search_tool = DuckDuckGoSearchTool()

llm_engine = HfApiEngine("Qwen/Qwen2.5-72B-Instruct")
# Ajanları her iki araçla başlatın.
agent = ReactCodeAgent(
    tools=[image_generation_tool, search_tool], llm_engine=llm_engine
)

# Çalıştır!
result = agent.run(
    "Generate me a photo of the car that James bond drove in the latest movie.",
)
result
```

![Image of an Aston Martin DB5](https://huggingface.co/datasets/huggingface/cookbook-images/resolve/main/agents_db5.png)

## 2. 📚💬 RAG ile yinelemeli sorgu iyileştirme ve kaynak seçimi

Kısa tanım: Retrieval-Augmented-Generation (RAG), ___“bir kullanıcı sorgusunu yanıtlamak için bir büyük dil modeli (LLM) kullanır, ancak yanıtı veri setinden elde edilen verilere dayandırır”___.

Bu yöntemin yalın ya da fine-tuned bir LLM kullanımına göre birçok avantajı vardır: bunlardan birkaçını saymak gerekirse, cevabı doğru gerçeklere dayandırmaya ve karışıklıkları azaltmaya izin verir, LLM'e özgü bilgileri sağlamaya ve bilgi tabanından veriye erişimin ince taneli kontrolüne izin verir.

- Şöyle bir senaryo düşünelim. RAG yöntemini uygulamak istiyoruz, ancak bazı parametrelerin dinamik olarak belirlenmesi gereken ek bir koşulumuz var. Örneğin, kullanıcı sorgusuna bağlı olarak aramayı bilgi tabanının belirli alt kümeleriyle sınırlamak isteyebiliriz veya alınan belge sayısını ayarlamak isteyebiliriz. Peki, **bu parametreleri kullanıcı sorgusuna göre nasıl dinamik olarak ayarlayabiliriz?**

- RAG'de sık karşılaşılan bir sorun, kullanıcı sorgusuna verilen cevabın hangi belgeden geldiğinin bulunmamasıdır. **Eğer önceki sonuçlar alakalı değilse, retriever'ı değiştirilmiş bir sorgu ile tekrar çalıştırarak sonuç alma şansımız var mı?**


🔧 Yukarıdaki noktaları basit bir şekilde çözebiliriz: **ajanımıza retriever'ın parametrelerinin kontrolünü vereceğiz!**

➡️ Hadi bunu nasıl yapacağımızı gösterelim. İlk olarak üzerinde RAG uygulamak istediğimiz bir bilgi tabanını yüklüyoruz: bu veri seti, markdown olarak depolanan birçok Hugging Face kütüphanesinin dokümantasyon sayfalarının bir derlemesidir.


```python
import datasets

knowledge_base = datasets.load_dataset("m-ric/huggingface_doc", split="train")
```

Şimdi veri setini işleyerek ve retriever tarafından kullanılacak bir vektör veritabanına depolayarak bilgi tabanını hazırlıyoruz. Vektör veritabanları için mükemmel yardımcı programlara sahip olduğu için LangChain'i kullanacağız:


```python
from langchain.docstore.document import Document
from langchain.text_splitter import RecursiveCharacterTextSplitter
from langchain.vectorstores import FAISS
from langchain_community.embeddings import HuggingFaceEmbeddings

source_docs = [
    Document(page_content=doc["text"], metadata={"source": doc["source"].split("/")[1]})
    for doc in knowledge_base
]

docs_processed = RecursiveCharacterTextSplitter(chunk_size=500).split_documents(
    source_docs
)[:1000]

embedding_model = HuggingFaceEmbeddings(model_name="thenlper/gte-small")
vectordb = FAISS.from_documents(documents=docs_processed, embedding=embedding_model)
```

Artık veritabanımız hazır olduğuna göre, kullanıcı sorgularını buna göre yanıtlayan bir RAG sistemi oluşturalım!

Sistemimizin sorguya bağlı olarak yalnızca en alakalı bilgi kaynaklarından seçim yapmasını istiyoruz.

Dökümantasyon sayfalarımız aşağıdaki kaynaklardan gelecek:

```python
>>> all_sources = list(set([doc.metadata["source"] for doc in docs_processed]))
>>> print(all_sources)
```

<pre>
['datasets-server', 'datasets', 'optimum', 'gradio', 'blog', 'course', 'hub-docs', 'pytorch-image-models', 'peft', 'evaluate', 'diffusers', 'hf-endpoints-documentation', 'deep-rl-class', 'transformers']
</pre>

👉 Şimdi ajanlarımızın bilgi tabanından bilgi almak için kullanabileceği bir `RetrieverTool` oluşturalım.

Vectordb'yi aracın bir özelliği olarak eklememiz gerektiği için, [basit araç oluşturusunu](https://huggingface.co/docs/transformers/main/en/agents#create-a-new-tool) ve `@tool` dekoratörünü kullanmak yeterli olmayacak. Bu nedenle, [gelişmiş ajanlar dökümantasyonunda](https://huggingface.co/docs/transformers/main/en/agents_advanced#directly-define-a-tool-by-subclassing-tool-and-share-it-to-the-hub)  belirtilen gelişmiş yapılandırmayı takip edeceğiz

```python
import json
from transformers.agents import Tool
from langchain_core.vectorstores import VectorStore


class RetrieverTool(Tool):
    name = "retriever"
    description = "Retrieves some documents from the knowledge base that have the closest embeddings to the input query."
    inputs = {
        "query": {
            "type": "string",
            "description": "The query to perform. This should be semantically close to your target documents. Use the affirmative form rather than a question.",
        },
        "source": {"type": "string", "description": ""},
        "number_of_documents": {
            "type": "string",
            "description": "the number of documents to retrieve. Stay under 10 to avoid drowning in docs",
        },
    }
    output_type = "string"

    def __init__(self, vectordb: VectorStore, all_sources: str, **kwargs):
        super().__init__(**kwargs)
        self.vectordb = vectordb
        self.inputs["source"]["description"] = (
            f"The source of the documents to search, as a str representation of a list. Possible values in the list are: {all_sources}. If this argument is not provided, all sources will be searched.".replace(
                "'", "`"
            )
        )

    def forward(self, query: str, source: str = None, number_of_documents=7) -> str:
        assert isinstance(query, str), "Your search query must be a string"
        number_of_documents = int(number_of_documents)

        if source:
            if isinstance(source, str) and "[" not in str(
                source
            ):  # eğer kaynak bir listeyi temsil etmiyorsa
                source = [source]
            source = json.loads(str(source).replace("'", '"'))

        docs = self.vectordb.similarity_search(
            query,
            filter=({"source": source} if source else None),
            k=number_of_documents,
        )

        if len(docs) == 0:
            return "No documents found with this filtering. Try removing the source filter."
        return "Retrieved documents:\n\n" + "\n===Document===\n".join(
            [doc.page_content for doc in docs]
        )
```

### Opsiyonel: Retriever aracınızı Hub'da paylaşın

Aracınızı Hub'da paylaşmak için, önce RetrieverTool tanım hücresindeki kodu kopyalayıp, örneğin `retriever.py` gibi bir adla yeni bir dosyaya yapıştırın.

Araç ayrı bir dosyadan yüklendiğinde, aşağıdaki kodu kullanarak Hub'a gönderebilirsiniz (`yazma` yetkisine sahip bir token ile giriş yaptığınızdan emin olun).

```python
share_to_hub = True

if share_to_hub:
    from huggingface_hub import login
    from retriever import RetrieverTool

    login("your_token")

    tool = RetrieverTool(vectordb, all_sources)

    tool.push_to_hub(repo_id="m-ric/retriever-tool")

    # Aracın Yüklenmesi
    from transformers.agents import load_tool

    retriever_tool = load_tool(
        "m-ric/retriever-tool", vectordb=vectordb, all_sources=all_sources
    )
```

### Ajanı çalıştırın!

```python
from transformers.agents import HfApiEngine, ReactJsonAgent

llm_engine = HfApiEngine("Qwen/Qwen2.5-72B-Instruct")

retriever_tool = RetrieverTool(vectordb=vectordb, all_sources=all_sources)
agent = ReactJsonAgent(tools=[retriever_tool], llm_engine=llm_engine, verbose=0)

agent_output = agent.run("Please show me a LORA finetuning script")

print("Final output:")
print(agent_output)
```

Peki burada ne oldu? İlk olarak, ajan belirli kaynaklarla (`['transformers', 'blog']`) birlikte retriever'ı başlattı.

Ancak bu arama yeterli sonucu vermedi. Sorun değil! Ajan, önceki sonuçlar üzerinde tekrar çalışabildiği için, daha az kısıtlayıcı arama parametreleriyle yeniden sorgulama yaptı ve sonuç olarak araştırma başarılı oldu!

Bir LLM ajanının bir retriever aracını kullanarak dinamik olarak sorguyu ve retrieval parametrelerini değiştirebilmesi, RAG'in daha genel bir formülasyonunu oluşturur ve aynı zamanda yinelemeli sorgu iyileştirme gibi birçok RAG geliştirme tekniğini de kapsar.

**Bir retriever'ı araç olarak kullanıp, sorguyu ve diğer** veri çekme parametrelerini dinamik olarak değiştirebilen **bir LLM ajanı kullanmak**, yinelemeli sorgu iyileştirmesi gibi birçok RAG geliştirme tekniğini de kapsayan **RAG'in daha genel bir formülasyonudur**.

## 3. 💻 Python Kodunda Hata Ayıklama
ReactCodeAgent'in yerleşik bir Python kod yorumlayıcısı olduğundan, hatalı Python scriptimizi debug etmek için kullanabiliriz!

```python
from transformers import ReactCodeAgent

agent = ReactCodeAgent(tools=[], llm_engine=HfApiEngine("Qwen/Qwen2.5-72B-Instruct"))

code = """
list=[0, 1, 2]

for i in range(4):
    print(list(i))
"""

final_answer = agent.run(
    "I have some code that creates a bug: please debug it, then run it to make sure it works and return the final code",
    code=code,
)
```

Gördüğünüz gibi, ajan verilen kodu deniyor, bir hata alıyor, hatayı analiz ediyor, kodu düzeltiyor ve çalıştığını gördükten sonra geri veriyor!

Sonuç olarak düzeltilen kodun son hali:

```python
>>> print(final_answer)
```

<pre>
my_list = [0, 1, 2]

for i in range(4):
    if i < len(my_list):
        print(my_list[i])
    else:
        print("Index out of range")
</pre>

## 4. Kendi LLM motorunuzu oluşturun (OpenAI)

Kendi LLM motorunuzu oluşturmak gerçekten çok kolay:
sadece bu kriterlere sahip bir `__call__` yöntemine ihtiyaç duyar:
1. Girdi olarak [ChatML formatında](https://huggingface.co/docs/transformers/main/en/chat_templating#introduction) bir mesaj listesi alır ve cevabı çıktı olarak verir.
2. `stop_sequences` argümanını destekleyerek metin üretmeyi durduracak dizileri tanımlar.
3. LLM'inizin desteklediği mesaj rolü (asistan, kullanıcı vb.) türlerine göre bazı mesaj rollerini dönüştürmeniz gerekebilir.

```python
import os
from openai import OpenAI
from transformers.agents.llm_engine import MessageRole, get_clean_message_list

openai_role_conversions = {
    MessageRole.TOOL_RESPONSE: "user",
}


class OpenAIEngine:
    def __init__(self, model_name="gpt-4o-2024-05-13"):
        self.model_name = model_name
        self.client = OpenAI(
            api_key=os.getenv("OPENAI_API_KEY"),
        )

    def __call__(self, messages, stop_sequences=[]):
        # Güvenli mesaj listesini edinin
        messages = get_clean_message_list(
            messages, role_conversions=openai_role_conversions
        )

        # LLM çıktısını alın
        response = self.client.chat.completions.create(
            model=self.model_name,
            messages=messages,
            stop=stop_sequences,
        )
        return response.choices[0].message.content


openai_engine = OpenAIEngine()
agent = ReactCodeAgent(llm_engine=openai_engine, tools=[])

code = """
list=[0, 1, 2]

for i in range(4):
    print(list(i))
"""

final_answer = agent.run(
    "I have some code that creates a bug: please debug it and return the final code",
    code=code,
)
```

```python
>>> print(final_answer)
```

<pre>
my_list = [0, 1, 2]  # Renamed the list to avoid using the built-in name

for i in range(len(my_list)):  # Changed the range to be within the length of the list
    print(my_list[i])  # Corrected the list access syntax
</pre>

## ➡️ Son olarak

Yukarıdaki kullanım örnekleri, Agents'ın sunduğu olanaklar hakkında size bir fikir verecektir.

Daha gelişmiş kullanım için [dökümantasyonu](https://huggingface.co/docs/transformers/en/transformers_agents) ve Llama-3-70B'yi temel alan ve son derece zorlayıcı GAIA Liderlik Tablosunda birçok GPT-4 ajanlarını geride bırakan kendi ajanımızı oluşturmamızı sağlayan [bu deneyi](https://github.com/aymeric-roucher/agent_reasoning_benchmark/blob/main/benchmark_gaia.ipynb) okuyun!

Tüm geri bildirimleri bekliyoruz, Agents'ı geliştirmemize yardımcı olacak!

<EditOnGithub source="https://github.com/huggingface/cookbook/blob/main/notebooks/tr/agents.md" />