# Transformersで何ができる?

<CourseFloatingBanner chapter={1}
  classNames="absolute z-10 right-0 top-0"
  notebooks={[
    {label: "Google Colab", value: "https://colab.research.google.com/github/huggingface/notebooks/blob/master/course/en/chapter1/section3.ipynb"},
    {label: "Aws Studio", value: "https://studiolab.sagemaker.aws/import/github/huggingface/notebooks/blob/master/course/en/chapter1/section3.ipynb"},
]} />


このセクションでは、Transformerモデルができることを見ていき、🤗 Transformersライブラリの最初のツールとして `pipeline()` 関数を使ってみましょう。

<Tip>
👀 右上に<em>Open in Colab</em>というボタンがありますよね？それをクリックすると、このセクションのすべてのコードサンプルを含むGoogle Colabノートブックが開きます。このボタンは、コードサンプルを含むどのセクションにも存在します。

ローカルでサンプルを実行したい場合は、<a href="/course/chapter0">セットアップ</a>を参照することをお勧めします。
</Tip>

## Transformersは至るところに!

Transformerモデルは前節で述べたようなあらゆる種類のNLPタスクを解決するために使用されています。ここでは、Hugging FaceとTransformerモデルを使用している企業や組織を紹介します。それらの組織はまた、モデルを共有することでコミュニティに還元しています。

<img src="https://huggingface.co/datasets/huggingface-course/documentation-images/resolve/main/en/chapter1/companies.PNG" alt="Companies using Hugging Face" width="100%">

[🤗 Transformers library](https://github.com/huggingface/transformers)は、それらの共有モデルを作成し、使用するための機能を提供します。[Model Hub](https://huggingface.co/models)には、誰でもダウンロードして使用できる何千もの事前学習済みモデルが含まれています。また、あなた自身のモデルをModel Hubにアップロードすることも可能です。

<Tip>
⚠️Hugging Face Hubはトランスフォーマーモデルに限定されるものではありません。誰でも好きな種類のモデルやデータセットを共有することができます!すべての利用可能な機能の恩恵を受けるために<a href="https://huggingface.co/join">huggingface.coのアカウントを作成</a>しましょう!
</Tip>  

<br>

Transformerのモデルがどのように機能するのかを知る前に、いくつかの興味深いNLPの問題を解決するため、Transformerがどのように使われるのか、いくつかの例で見ていきましょう。

## pipelineを使ってタスクを実行する

<Youtube id="tiZFewofSLM" />

🤗 Transformers ライブラリの中で最も基本的なオブジェクトは `pipeline()` 関数です。これはモデルを必要な前処理と後処理のステップに接続し、任意のテキストを直接入力して理解しやすい答えを得ることを可能にします。

```python
from transformers import pipeline

classifier = pipeline("sentiment-analysis")
classifier("I've been waiting for a HuggingFace course my whole life.")
```

```python out
[{'label': 'POSITIVE', 'score': 0.9598047137260437}]
```

複数の文章を入力することも可能です。

```python
classifier(
    ["I've been waiting for a HuggingFace course my whole life.", "I hate this so much!"]
)
```

```python out
[{'label': 'POSITIVE', 'score': 0.9598047137260437},
 {'label': 'NEGATIVE', 'score': 0.9994558095932007}]
```

デフォルトでは、このpipelineは英語の感情分析用にファインチューニングされた特定の事前学習モデルを使用します。このモデルは `classifier` オブジェクトを作成する際にダウンロードされ、キャッシュされます。コマンドを再実行すると、キャッシュされたモデルが代わりに使用され、モデルを再度ダウンロードする必要はありません。

pipelineにテキストを渡す場合、主に3つのステップがあります。

1. テキストはモデルが理解できる形式に前処理される。
2. 前処理された入力がモデルに渡される。
3. 予測結果を理解できるように、モデルの後処理が行われる。

現在[利用可能なpipeline](https://huggingface.co/transformers/main_classes/pipelines.html)の一部を紹介します。

- `feature-extraction` (テキストのベクトル表現を取得)
- `fill-mask`
- `ner` (固有表現認識)
- `question-answering`
- `sentiment-analysis`
- `summarization`
- `text-generation`
- `translation`
- `zero-shot-classification`

では、いくつか見ていきましょう！

## ゼロショット分類

まず、ラベル付けされていないテキストを分類する必要があるような、より困難なタスクに取り組むことから始めます。これは実際のプロジェクトでよくあるシナリオです。なぜなら、テキストにアノテーションをつけるのは通常時間がかかり、専門知識が必要だからです。このような場合、`zero-shot-classification` pipelineは非常に強力です。分類に使用するラベルを指定できるので、事前に学習したモデルのラベルに依存する必要がありません。肯定的か否定的かの2つのラベルを使って、モデルがどのようにテキストを分類するかは既に見たとおりです。しかし、他の任意のラベルセットを使ってテキストを分類することもできます。


```python
from transformers import pipeline

classifier = pipeline("zero-shot-classification")
classifier(
    "This is a course about the Transformers library",
    candidate_labels=["education", "politics", "business"],
)
```

```python out
{'sequence': 'This is a course about the Transformers library',
 'labels': ['education', 'business', 'politics'],
 'scores': [0.8445963859558105, 0.111976258456707, 0.043427448719739914]}
```

このpipelineは _zero-shot_ と呼ばれます。なぜなら、これを使うために自前のデータセットでモデルのファインチューニングをする必要がないからです。任意のラベルのリストに対して直接確率スコアを返すことができます。

<Tip>

✏️ **試してみよう!** 独自の入力とラベルで遊んでみて、モデルがどのように振る舞うか見てみましょう。

</Tip>

## 文章生成

では、pipelineを使ってテキストを生成する方法を見てみましょう。主なアイデアは、プロンプトを与えると、モデルが残りのテキストを生成してそれを補完することです。これは、多くの携帯電話に搭載されている予測入力機能に類似しています。テキスト生成にはランダム性が含まれるため、以下と同様な結果が得られないのが通常です。

```python
from transformers import pipeline

generator = pipeline("text-generation")
generator("In this course, we will teach you how to")
```

```python out
[{'generated_text': 'In this course, we will teach you how to understand and use '
                    'data flow and data interchange when handling user data. We '
                    'will be working with one or more of the most commonly used '
                    'data flows — data flows of various types, as seen by the '
                    'HTTP'}]
```

引数`num_return_sequences`で異なるシーケンスの生成数を、引数`max_length`で出力テキストの合計の長さを制御することができます。

<Tip>


✏️ **試してみよう!** `num_return_sequences` と `max_length` 引数を用いて、15語ずつの2つの文を生成してみましょう！

</Tip>


## pipelineでHubから任意のモデルを使用する

これまでの例では、タスクに応じたデフォルトのモデルを使用しましたが、特定のタスク（例えばテキスト生成）のpipelineで使用するモデルをHubから選択することも可能です。[Model Hub](https://huggingface.co/models)にアクセスし、左側の対応するタグをクリックすると、そのタスクでサポートされているモデルのみが表示されます。[このようなページ](https://huggingface.co/models?pipeline_tag=text-generation)が表示されるはずです。

それでは、[`distilgpt2`](https://huggingface.co/distilgpt2)モデルを試してみましょう! 先ほどと同じpipelineでロードする方法を説明します。

```python
from transformers import pipeline

generator = pipeline("text-generation", model="distilgpt2")
generator(
    "In this course, we will teach you how to",
    max_length=30,
    num_return_sequences=2,
)
```

```python out
[{'generated_text': 'In this course, we will teach you how to manipulate the world and '
                    'move your mental and physical capabilities to your advantage.'},
 {'generated_text': 'In this course, we will teach you how to become an expert and '
                    'practice realtime, and with a hands on experience on both real '
                    'time and real'}]
```

言語タグをクリックして検索するモデルを絞り込み、他の言語でテキストを生成するモデルを選ぶことができます。Model Hubには、複数の言語をサポートする多言語モデルのチェックポイントもあります。

モデルをクリックで選択すると、オンラインで直接試用できるウィジェットが表示されます。このようにして、ダウンロードする前にモデルの機能をすばやくテストすることができます。

<Tip>

✏️ **試してみよう!** フィルターを使って、他の言語のテキスト生成モデルを探してみましょう。ウィジェットで自由に遊んだり、pipelineで使ってみてください！

</Tip>


### 推論API

すべてのモデルは、Hugging Face [ウェブサイト](https://huggingface.co/)で公開されているInference APIを使って、ブラウザから直接テストすることが可能です。このページでは、任意の文字列を入力し、モデルが入力データを処理する様子を見ることで、直接モデルで遊ぶことができます。

このウィジェットを動かすInference APIは、有料製品としても提供されており、ワークフローに必要な場合は便利です。詳しくは[価格ページ](https://huggingface.co/pricing)をご覧ください。

## 空所穴埋め

次に試すpipelineは`fill-mask`です。このタスクのアイデアは、与えられたテキストの空白を埋めることです。

```python
from transformers import pipeline

unmasker = pipeline("fill-mask")
unmasker("This course will teach you all about <mask> models.", top_k=2)
```

```python out
[{'sequence': 'This course will teach you all about mathematical models.',
  'score': 0.19619831442832947,
  'token': 30412,
  'token_str': ' mathematical'},
 {'sequence': 'This course will teach you all about computational models.',
  'score': 0.04052725434303284,
  'token': 38163,
  'token_str': ' computational'}]
```

`top_k` 引数は、いくつの可能性を表示させたいかをコントロールします。ここでは、モデルが特別な `<mask>` という単語を埋めていることに注意してください。これはしばしば *mask token* と呼ばれます。他の空所穴埋めモデルは異なるマスクトークンを持つかもしれないので、他のモデルを探索するときには常に適切なマスクワードを確認するのが良いでしょう。それを確認する1つの方法は、ウィジェットで使用されているマスクワードを見ることです。

<Tip>

✏️ **試してみよう!** Hub で `bert-base-cased` モデルを検索し、推論API ウィジェットでそのマスクワードを特定します。このモデルは上記の `pipeline` の例文に対して何を予測するでしょうか？

</Tip>

## 固有表現認識

固有表現認識（NER）は、入力されたテキストのどの部分が人物、場所、組織などの固有表現に対応するかをモデルが見つけ出すタスクです。例を見てみましょう。

```python
from transformers import pipeline

ner = pipeline("ner", grouped_entities=True)
ner("My name is Sylvain and I work at Hugging Face in Brooklyn.")
```

```python out
[{'entity_group': 'PER', 'score': 0.99816, 'word': 'Sylvain', 'start': 11, 'end': 18}, 
 {'entity_group': 'ORG', 'score': 0.97960, 'word': 'Hugging Face', 'start': 33, 'end': 45}, 
 {'entity_group': 'LOC', 'score': 0.99321, 'word': 'Brooklyn', 'start': 49, 'end': 57}
]
```

ここでは、モデルはSylvainが人（PER）、Hugging Faceが組織（ORG）、Brooklynが場所（LOC）であることを正しく識別しています。

pipelineの作成機能でオプション `grouped_entities=True` を渡すと、同じエンティティに対応する文の部分を再グループ化するようpipelineに指示します。ここでは、名前が複数の単語で構成されていても、モデルは "Hugging" と "Face" を一つの組織として正しくグループ化しています。実際、次の章で説明するように、前処理ではいくつかの単語をより小さなパーツに分割することさえあります。例えば、`Sylvain`は4つの部分に分割されます。`S`, `##yl`, `##va`, and `##in`.です。後処理の段階で、pipelineはこれらの断片をうまく再グループ化しました。

<Tip>

✏️ **試してみよう!** Model Hubで英語の品詞タグ付け（通常POSと略される）を行えるモデルを検索してください。このモデルは、上の例の文に対して何を予測するでしょうか？

</Tip>

## 質問応答

質問応答pipelineは、与えられた文脈から得た情報を使って質問に答えます。

```python
from transformers import pipeline

question_answerer = pipeline("question-answering")
question_answerer(
    question="Where do I work?",
    context="My name is Sylvain and I work at Hugging Face in Brooklyn",
)
```

```python out
{'score': 0.6385916471481323, 'start': 33, 'end': 45, 'answer': 'Hugging Face'}
```

このpipelineは、提供されたコンテキストから情報を抽出することで動作し、答えを生成するわけではないことに注意してください。

## 要約

要約とは、文章中の重要な部分をすべて（あるいはほとんど）維持したまま、より短い文章にするタスクです。以下はその例です。

```python
from transformers import pipeline

summarizer = pipeline("summarization")
summarizer(
    """
    America has changed dramatically during recent years. Not only has the number of 
    graduates in traditional engineering disciplines such as mechanical, civil, 
    electrical, chemical, and aeronautical engineering declined, but in most of 
    the premier American universities engineering curricula now concentrate on 
    and encourage largely the study of engineering science. As a result, there 
    are declining offerings in engineering subjects dealing with infrastructure, 
    the environment, and related issues, and greater concentration on high 
    technology subjects, largely supporting increasingly complex scientific 
    developments. While the latter is important, it should not be at the expense 
    of more traditional engineering.
    Rapidly developing economies such as China and India, as well as other 
    industrial countries in Europe and Asia, continue to encourage and advance 
    the teaching of engineering. Both China and India, respectively, graduate 
    six and eight times as many traditional engineers as does the United States. 
    Other industrial countries at minimum maintain their output, while America 
    suffers an increasingly serious decline in the number of engineering graduates 
    and a lack of well-educated engineers.
"""
)
```

```python out
[{'summary_text': ' America has changed dramatically during recent years . The '
                  'number of engineering graduates in the U.S. has declined in '
                  'traditional engineering disciplines such as mechanical, civil '
                  ', electrical, chemical, and aeronautical engineering . Rapidly '
                  'developing economies such as China and India, as well as other '
                  'industrial countries in Europe and Asia, continue to encourage '
                  'and advance engineering .'}]
```

テキスト生成と同様に、結果に対して `max_length` や `min_length` を指定することができます。


## 翻訳

翻訳の場合、タスク名に言語ペアを指定すれば（`"translation_en_to_fr"`など）デフォルトのモデルを使うこともできますが、一番簡単なのは [Model Hub](https://huggingface.co/models) で使いたいモデルを選ぶことです。ここでは、フランス語から英語への翻訳を試してみます。

```python
from transformers import pipeline

translator = pipeline("translation", model="Helsinki-NLP/opus-mt-fr-en")
translator("Ce cours est produit par Hugging Face.")
```

```python out
[{'translation_text': 'This course is produced by Hugging Face.'}]
```

テキスト生成や要約と同様に、結果に対して `max_length` や `min_length` を指定することができます。

<Tip>

✏️ **試してみよう!** 他言語の翻訳モデルを検索して、前の文章をいくつかの異なる言語に翻訳してみましょう。

</Tip>

これまで紹介したpipelineは、ほとんどがデモンストレーションのためのものです。これらは特定のタスクのためにプログラムされたものであり、それらのバリエーションを実行することはできません。次の章では、`pipeline()`関数の中身と、その動作をカスタマイズする方法を学びます。


<EditOnGithub source="https://github.com/huggingface/course/blob/main/chapters/ja/chapter1/3.mdx" />