# Compartilhando modelos pré-treinados

{#if fw === 'pt'}

<CourseFloatingBanner chapter={4}
  classNames="absolute z-10 right-0 top-0"
  notebooks={[
    {label: "Google Colab", value: "https://colab.research.google.com/github/huggingface/notebooks/blob/master/course/pt/chapter4/section3_pt.ipynb"},
    {label: "Aws Studio", value: "https://studiolab.sagemaker.aws/import/github/huggingface/notebooks/blob/master/course/pt/chapter4/section3_pt.ipynb"},
]} />

{:else}

<CourseFloatingBanner chapter={4}
  classNames="absolute z-10 right-0 top-0"
  notebooks={[
    {label: "Google Colab", value: "https://colab.research.google.com/github/huggingface/notebooks/blob/master/course/pt/chapter4/section3_tf.ipynb"},
    {label: "Aws Studio", value: "https://studiolab.sagemaker.aws/import/github/huggingface/notebooks/blob/master/course/pt/chapter4/section3_tf.ipynb"},
]} />

{/if}

Nas etapas abaixo, veremos as maneiras mais fáceis de compartilhar modelos pré-treinados para o Hub 🤗. Há ferramentas e utilitários disponíveis que facilitam o compartilhamento e atualização de modelos diretamente no Hub, que exploraremos a seguir.

<Youtube id="9yY3RB_GSPM"/>

Encorajamos todos os usuários que treinam modelos a contribuir compartilhando-os com a comunidade - compartilhar modelos, mesmo quando treinados em conjuntos de dados muito específicos, ajudará outros, economizando tempo e recursos e fornecendo acesso a artefatos úteis treinados. Por sua vez, você pode se beneficiar do trabalho que outros realizaram!

Há três maneiras de se criar novos repositórios modelo:

- Usando a API`push_to_hub`
- Usando a biblioteca Python `huggingface_hub`
- Usando a interface web 

Uma vez criado um repositório, você pode fazer o upload de arquivos para ele via git e git-lfs. Nós o acompanharemos na criação de repositórios modelo e no upload de arquivos para eles nas seções seguintes.


## Usando a API`push_to_hub`

{#if fw === 'pt'}

<Youtube id="Zh0FfmVrKX0"/>

{:else}

<Youtube id="pUh5cGmNV8Y"/>

{/if}

A maneira mais simples de carregar arquivos no Hub é usando a API `push_to_hub`.

Antes de ir adiante, você precisará gerar um token de autenticação para que a API `huggingface_hub` saiba quem você é e a que namespaces você tem acesso de escrita. Certifique-se de estar em um ambiente onde você tenha `transformers` instalado (ver [Setup](/course/chapter0)). Se você estiver em um notebook, você pode utilizar a seguinte função para fazer o login:


```python
from huggingface_hub import notebook_login

notebook_login()
```

Em um terminal, você pode rodar:

```bash
huggingface-cli login
```

Em ambos os casos, você será solicitado seu nome de usuário e senha, que são os mesmos que você usa para fazer o login no Hub. Se você ainda não tem um perfil do Hub, você deve criar um [aqui](https://huggingface.co/join).

Ótimo! Agora você tem seu token de autenticação armazenado em sua pasta de cache. Vamos criar alguns repositórios!

{#if fw === 'pt'}

Se você usou a API do `Trainer` para treinar um modelo, a maneira mais fácil de carregá-lo no Hub é definir `push_to_hub=True` quando você definir seus `TrainingArguments`:

```py
from transformers import TrainingArguments

training_args = TrainingArguments(
    "bert-finetuned-mrpc", save_strategy="epoch", push_to_hub=True
)
```

Quando você chama `trainer.train()`, o `Trainer` então carregará seu modelo no Hub cada vez que ele for salvo (aqui a cada época) em um repositório em seu namespace. Esse repositório será nomeado como o diretório de saída que você escolheu (aqui `bert-finetuned-mrpc`), mas você pode escolher um nome diferente com `hub_model_id = "a_diferent_name"`.

Para enviar seu modelo para uma organização da qual você é membro, basta passá-lo com `hub_model_id = "my_organization/my_repo_name"`.

Uma vez terminado seu treinamento, você deve fazer um último `trainer.push_to_hub()` para carregar a última versão de seu modelo. Ele também gerará um cartão modelo com todos os metadados relevantes, relatando os hiperparâmetros utilizados e os resultados da avaliação! Aqui está um exemplo do conteúdo que você pode encontrar em um cartão modelo deste tipo:

<div class="flex justify-center">
  <img src="https://huggingface.co/datasets/huggingface-course/documentation-images/resolve/main/en/chapter4/model_card.png" alt="An example of an auto-generated model card." width="100%"/>
</div>

{:else}


Se você estiver utilizando Keras para treinar seu modelo, a maneira mais fácil de carregá-lo no Hub é passar um `PushToHubCallback` quando você chama de `model.fit()`:

```py
from transformers import PushToHubCallback

callback = PushToHubCallback(
    "bert-finetuned-mrpc", save_strategy="epoch", tokenizer=tokenizer
)
```

Então você deve adicionar `callbacks=[callback]` em sua chamada de `model.fit()`. A chamada de retorno será então enviada ao Hub cada vez que o modelo for salvo (aqui a cada época) em um repositório em seu namespace. Esse repositório será nomeado como o diretório de saída que você escolheu (aqui `bert-finetuned-mrpc`), mas você pode escolher um nome diferente com `hub_model_id = "a_diferent_name"`.

Para enviar seu modelo para uma organização da qual você é membro, basta passá-lo com `hub_model_id = "my_organization/my_repo_name"`.

{/if}

Em um nível inferior, o acesso ao Model Hub pode ser feito diretamente nos modelos, tokenizers e objetos de configuração através de seu método `push_to_hub()`. Este método cuida da criação do repositório e empurra os arquivos modelo e tokenizer diretamente para o repositório. Não é necessário nenhum tratamento manual, ao contrário do que acontece com a API, veremos abaixo.

Para se ter uma idéia de como funciona, vamos primeiro inicializar um modelo e um tokenizer:

{#if fw === 'pt'}
```py
from transformers import AutoModelForMaskedLM, AutoTokenizer

checkpoint = "camembert-base"

model = AutoModelForMaskedLM.from_pretrained(checkpoint)
tokenizer = AutoTokenizer.from_pretrained(checkpoint)
```
{:else}
```py
from transformers import TFAutoModelForMaskedLM, AutoTokenizer

checkpoint = "camembert-base"

model = TFAutoModelForMaskedLM.from_pretrained(checkpoint)
tokenizer = AutoTokenizer.from_pretrained(checkpoint)
```
{/if}

Você é livre para fazer o que quiser com elas - adicionar fichas ao tokenizer, treinar o modelo, afinar o modelo. Quando você estiver satisfeito com o modelo, pesos e tokenizer resultantes, você pode aproveitar o método `push_to_hub()` diretamente disponível no objeto `model`:

```py
model.push_to_hub("dummy-model")
```

Isto criará o novo repositório `dummy-model`  em seu perfil, e o preencherá com seus arquivos de modelos.
Faça o mesmo com o tokenizer, para que todos os arquivos estejam agora disponíveis neste repositório:

```py
tokenizer.push_to_hub("dummy-model")
```

Se você pertence a uma organização, basta especificar o argumento `organization` a ser carregado no namespace dessa organização:

```py
tokenizer.push_to_hub("dummy-model", organization="huggingface")
```

Se você desejar utilizar um toke específica do Hugging Face, você é livre para especificá-la também para o método `push_to_hub()`:

```py
tokenizer.push_to_hub("dummy-model", organization="huggingface", use_auth_token="<TOKEN>")
```

Agora vá até o Model Hub para encontrar seu modelo recém-carregado: *https://huggingface.co/user-or-organization/dummy-model*.

Clique na aba "Files and versions", e você deve ver os arquivos visíveis na seguinte captura de tela:

{#if fw === 'pt'}
<div class="flex justify-center">
<img src="https://huggingface.co/datasets/huggingface-course/documentation-images/resolve/main/en/chapter4/push_to_hub_dummy_model.png" alt="Dummy model containing both the tokenizer and model files." width="80%"/>
</div>
{:else}
<div class="flex justify-center">
<img src="https://huggingface.co/datasets/huggingface-course/documentation-images/resolve/main/en/chapter4/push_to_hub_dummy_model_tf.png" alt="Dummy model containing both the tokenizer and model files." width="80%"/>
</div>
{/if}

<Tip>

✏️ **Teste-o!** Pegue o modelo e o tokenizer associados ao checkpoint `bert-base-cased` e carregue-os para um repo em seu namespace utilizando o método `push_to_hub()`. Verifique novamente se o repo aparece corretamente em sua página antes de excluí-lo.

</Tip>

Como você já viu, o método `push_to_hub()` aceita vários argumentos, tornando possível carregar para um repositório específico ou espaço de nomes de organizações, ou utilizar um token API diferente. Recomendamos que você dê uma olhada na especificação do método disponível diretamente na documentação [🤗 Transformers documentation](https://huggingface.co/transformers/model_sharing.html) para ter uma idéia do que é possível.

O método `push_to_hub()` é apoiado pelo pacote [`huggingface_hub`](https://github.com/huggingface/huggingface_hub) Python, que oferece uma API direta para o Hub Hugging Face. Está integrado ao 🤗 Transformers e várias outras bibliotecas de aprendizagem de máquinas, como [`allenlp`](https://github.com/allenai/allennlp). Embora nos concentremos na integração do 🤗 Transformers neste capítulo, integrá-lo em seu próprio código ou biblioteca é simples.

Salte para a última seção para ver como carregar arquivos em seu repositório recém-criado!

## Usando a biblioteca Python `huggingface_hub`

A biblioteca Python`huggingface_hub` é um pacote que oferece um conjunto de ferramentas para os hubs do modelo e dos conjuntos de dados. Ela fornece métodos e classes simples para tarefas comuns como obter informações sobre os repositórios no centro e gerenciá-los. Ele fornece APIs simples que funcionam em cima do git para gerenciar o conteúdo desses repositórios e para integrar o Hub em seus projetos e bibliotecas.

Da mesma forma que a utilização da API `push_to_hub`, isto exigirá que você tenha seu token API salvo em seu cache. Para fazer isso, você precisará utilizar o comando `login` do CLI, como mencionado na seção anterior (mais uma vez, certifique-se de utilizar antes desses comandos o caracter `!` se estiver rodando no Google Colab):

```bash
huggingface-cli login
```

O pacote `huggingface_hub` oferece vários métodos e classes que são úteis para nosso propósito. Em primeiro lugar, existem alguns métodos para gerenciar a criação de repositórios, exclusão, e outros:

```python no-format
from huggingface_hub import (
    # Gestão de usuários
    login,
    logout,
    whoami,

    # Criação e gestão de repositório
    create_repo,
    delete_repo,
    update_repo_visibility,

    #E alguns métodos para recuperar/trocar informações sobre o conteúdo
    list_models,
    list_datasets,
    list_metrics,
    list_repo_files,
    upload_file,
    delete_file,
)
```


Além disso, oferece uma poderosa classe `Repository` para gerenciar um repositório local. Vamos explorar esses métodos e essa classe na próxima seção para entender como aproveitá-los.

O método `create_repo` pode ser utilizado para criar um novo repositório no centro:

```py
from huggingface_hub import create_repo

create_repo("dummy-model")
```

Isto criará o repositório `dummy-model` em seu namespace. Se desejar, você pode especificar a que organização o repositório deve pertencer utilizando o argumento `organization`:

```py
from huggingface_hub import create_repo

create_repo("dummy-model", organization="huggingface")
```


Isto criará o repositório `dummy-model` no espaço de nomes `huggingface`, assumindo que você pertença a essa organização.
Outros argumentos que podem ser úteis são:

- `private`, a fim de especificar se o repositório deve ser visível de outros ou não.
- `token`,se você gostaria de substituir o token armazenada em seu cache por uma determinada token.
- `repo_type`, se você gostaria de criar um "`dataset` ou um "espaço" em vez de um modelo. Os valores aceitos são `"dataset"`  e `"space"`.

Uma vez criado o repositório, devemos adicionar arquivos a ele! Salte para a próxima seção para ver as três maneiras como isto pode ser tratado.


## Usando a interface web 

A interface web oferece ferramentas para gerenciar os repositórios diretamente no Hub. Usando a interface, você pode facilmente criar repositórios, adicionar arquivos (mesmo grandes!), explorar modelos, visualizar diffs, e muito mais.

Para criar um novo repositório, visite [huggingface.co/novo](https://huggingface.co/new):

<div class="flex justify-center">
<img src="https://huggingface.co/datasets/huggingface-course/documentation-images/resolve/main/en/chapter4/new_model.png" alt="Page showcasing the model used for the creation of a new model repository." width="80%"/>
</div>

Primeiro, especifique o proprietário do repositório: este pode ser você ou qualquer uma das organizações às quais você está afiliado. Se você escolher uma organização, o modelo será apresentado na página da organização e cada membro da organização terá a capacidade de contribuir com o repositório.

A seguir, digite o nome do seu modelo. Este também será o nome do repositório. Finalmente, você pode especificar se deseja que seu modelo seja público ou privado. Os modelos privados não podem ser encontrados publicamente.

Depois de criar seu repositório de modelos, você deve ver uma página como esta:

<div class="flex justify-center">
<img src="https://huggingface.co/datasets/huggingface-course/documentation-images/resolve/main/en/chapter4/empty_model.png" alt="An empty model page after creating a new repository." width="80%"/>
</div>

Aqui é onde seu modelo será hospedado. Para começar a preenchê-lo, você pode adicionar um arquivo README diretamente da interface web.

<div class="flex justify-center">
<img src="https://huggingface.co/datasets/huggingface-course/documentation-images/resolve/main/en/chapter4/dummy_model.png" alt="The README file showing the Markdown capabilities." width="80%"/>
</div>

O arquivo README está em Markdown - sinta-se à vontade para ficar louco com ele! A terceira parte deste capítulo é dedicada à construção de um modelo de cartão. Estes são de extrema importância para trazer valor ao seu modelo, pois estão onde você diz aos outros o que ele pode fazer.

Se você olhar a aba "Files and versions", você verá que ainda não há muitos arquivos - apenas o *README.md* que você acabou de criar e o arquivo *.gitattributes* que mantém o controle de arquivos grandes.

<div class="flex justify-center">
<img src="https://huggingface.co/datasets/huggingface-course/documentation-images/resolve/main/en/chapter4/files.png" alt="The 'Files and versions' tab only shows the .gitattributes and README.md files." width="80%"/>
</div>

A seguir, veremos como adicionar alguns novos arquivos.

## Fazendo upload dos arquivos de modelos

O sistema para gerenciar arquivos no Hub Hugging Face Hub é baseado no git para arquivos regulares, e git-lfs (que significa [Git Large File Storage](https://git-lfs.github.com/)) para arquivos maiores. 

Na seção seguinte, passamos por três maneiras diferentes de carregar arquivos no Hub: através de `huggingface_hub` e através de comandos de git.

### A abordagem: `upload_file`

A utilização do `upload_file` não requer que git e git-lfs sejam instalados em seu sistema. Ele empurra os arquivos diretamente para o Hub 🤗 utilizando solicitações HTTP POST. Uma limitação desta abordagem é que ele não lida com arquivos maiores que 5GB de tamanho.
Se seus arquivos forem maiores que 5GB, por favor, siga os dois outros métodos detalhados abaixo.

A API pode ser usada da seguinte forma:

```py
from huggingface_hub import upload_file

upload_file(
    "<path_to_file>/config.json",
    path_in_repo="config.json",
    repo_id="<namespace>/dummy-model",
)
```

Isto fará o upload do arquivo `config.json` disponível em `<path_to_file>` para a raiz do repositório como `config.json`, para o repositório `dummy-model`.
Outros argumentos que podem ser úteis são:

- `token`, se você gostaria de substituir o token armazenado em seu cache por um determinado token.
- `repo_type`, se você gostaria de carregar em um `dataset` ou em um `espaço` em vez de um modelo. Os valores aceitos são `"dataset"` e `"space"`.

### A classe: `Repository`

A classe `Repository` gerencia um repositório local de forma idiota. Ele resume a maioria dos pontos de dor que se pode ter com o git para fornecer todas as características que necessitamos. 

A utilização desta classe requer ter git e git-lfs instalados, portanto certifique-se de ter o git-lfs instalado (veja [aqui](https://git-lfs.github.com/) para instruções de instalação) e configure-o antes de começar. 

Para começar a brincar com o repositório que acabamos de criar, podemos começar inicializando-o em uma pasta local através da clonagem do repositório remoto:

```py
from huggingface_hub import Repository

repo = Repository("<path_to_dummy_folder>", clone_from="<namespace>/dummy-model")
```

Isto criou a pasta `<path_to_dummy_folder>` em nosso diretório de trabalho. Esta pasta contém apenas o arquivo `.gitattributes`, pois este é o único arquivo criado ao instanciar o repositório através do `create_repo`.

A partir deste ponto, podemos aproveitar vários dos métodos tradicionais do gitattributes:

```py
repo.git_pull()
repo.git_add()
repo.git_commit()
repo.git_push()
repo.git_tag()
```

E outros! Recomendamos dar uma olhada na documentação `Repository` disponível [aqui](https://github.com/huggingface/huggingface_hub/tree/main/src/huggingface_hub#advanced-programmatic-repository-management) para uma visão geral de todos os métodos disponíveis.

No momento, temos um modelo e um tokenizer que gostaríamos de empurrar para o centro. Clonamos com sucesso o repositório, portanto, podemos salvar os arquivos dentro desse repositório.

Primeiro nos certificamos de que nosso clone local esteja atualizado, puxando as últimas mudanças:

```py
repo.git_pull()
```

Uma vez feito isso, salvamos os arquivos do modelo e do tokenizer:

```py
model.save_pretrained("<path_to_dummy_folder>")
tokenizer.save_pretrained("<path_to_dummy_folder>")
```

O `<path_to_dummy_folder>` agora contém todos os modelos e arquivos de fichas. Seguimos o fluxo de trabalho habitual do git, adicionando arquivos à área de encenação, comprometendo-os e empurrando-os para o centro:

```py
repo.git_add()
repo.git_commit("Add model and tokenizer files")
repo.git_push()
```

Parabéns! Você acabou de empurrar seus primeiros arquivos para o centro.

### A abordagem: `baseada em git`

Esta é a própria abordagem do barebone para carregar arquivos: faremos isso com git e git-lfs diretamente. A maior parte da dificuldade é abstraída por abordagens anteriores, mas há algumas advertências com o seguinte método, então seguiremos um caso de uso mais complexo.

O uso desta classe requer ter git e git-lfs instalados, portanto, certifique-se de ter [git-lfs](https://git-lfs.github.com/) instalado (veja aqui as instruções de instalação) e configurado antes de começar. 

Primeiro comece inicializando o git-lfs:

```bash
git lfs install
```

```bash
Updated git hooks.
Git LFS initialized.
```

Uma vez feito isso, o primeiro passo é clonar seu repositório modelo:

```bash
git clone https://huggingface.co/<namespace>/<your-model-id>
```

Meu nome de usuário é `lysandre` e já utilizei o nome modelo `dummy`, então para mim o comando acaba parecendo o seguinte:

```
git clone https://huggingface.co/lysandre/dummy
```

Agora tenho uma pasta com o nome *dummy* em meu diretório de trabalho. Eu posso `cd` dentro da pasta e dar uma olhada no conteúdo:

```bash
cd dummy && ls
```

```bash
README.md
```

Se você acabou de criar seu repositório utilizando o método `create_repo` do Hugging Face Hub, esta pasta deve conter apenas um arquivo oculto `.gitattributes`. Se você seguiu as instruções da seção anterior para criar um repositório utilizando a interface web, a pasta deve conter um único arquivo *README.md* ao lado do arquivo oculto `.gitattributes`, como mostrado aqui.

Adicionar um arquivo de tamanho normal, como um arquivo de configuração, um arquivo de vocabulário, ou basicamente qualquer arquivo sob alguns megabytes, é feito exatamente como se faria em qualquer sistema baseado no gitattributes. Entretanto, arquivos maiores devem ser registrados através do git-lfs a fim de empurrá-los para *huggingface.co*. 

Vamos voltar a Python para gerar um modelo e tokenizer que gostaríamos de comprometer com nosso repositório dummy:

{#if fw === 'pt'}
```py
from transformers import AutoModelForMaskedLM, AutoTokenizer

checkpoint = "camembert-base"

model = AutoModelForMaskedLM.from_pretrained(checkpoint)
tokenizer = AutoTokenizer.from_pretrained(checkpoint)

# Do whatever with the model, train it, fine-tune it...

model.save_pretrained("<path_to_dummy_folder>")
tokenizer.save_pretrained("<path_to_dummy_folder>")
```
{:else}
```py
from transformers import TFAutoModelForMaskedLM, AutoTokenizer

checkpoint = "camembert-base"

model = TFAutoModelForMaskedLM.from_pretrained(checkpoint)
tokenizer = AutoTokenizer.from_pretrained(checkpoint)

# Do whatever with the model, train it, fine-tune it...

model.save_pretrained("<path_to_dummy_folder>")
tokenizer.save_pretrained("<path_to_dummy_folder>")
```
{/if}

Agora que salvamos alguns artefatos de modelo e tokenizer, vamos dar outra olhada na pasta *dummy*:

```bash
ls
```

{#if fw === 'pt'}
```bash
config.json  pytorch_model.bin  README.md  sentencepiece.bpe.model  special_tokens_map.json tokenizer_config.json  tokenizer.json
```

Se você olhar para os tamanhos de arquivo (por exemplo, com `ls -lh`), você deve ver que o arquivo de estado do modelo (*pytorch_model.bin*) é o único outlier, com mais de 400 MB.

{:else}
```bash
config.json  README.md  sentencepiece.bpe.model  special_tokens_map.json  tf_model.h5  tokenizer_config.json  tokenizer.json
```

Se você olhar para os tamanhos de arquivo (por exemplo, com `ls -lh`), você deve ver que o arquivo de estado do modelo (*t5_model.h5*) é o único outlier, com mais de 400 MB.

{/if}

<Tip>
✏️ Ao criar o repositório a partir da interface web, o arquivo *.gitattributes* é automaticamente configurado para considerar arquivos com certas extensões, como *.bin* e *.h5*, como arquivos grandes, e o git-lfs os rastreará sem nenhuma configuração necessária em seu lado.
</Tip> 

Agora podemos ir em frente e proceder como normalmente faríamos com os repositórios tradicionais da Git. Podemos adicionar todos os arquivos ao ambiente de encenação do Git utilizando o comando `git add`:

```bash
git add .
```

Podemos, então, dar uma olhada nos arquivos que estão atualmente em fase de montagem:

```bash
git status
```

{#if fw === 'pt'}
```bash
On branch main
Your branch is up to date with 'origin/main'.

Changes to be committed:
  (use "git restore --staged <file>..." to unstage)
  modified:   .gitattributes
	new file:   config.json
	new file:   pytorch_model.bin
	new file:   sentencepiece.bpe.model
	new file:   special_tokens_map.json
	new file:   tokenizer.json
	new file:   tokenizer_config.json
```
{:else}
```bash
On branch main
Your branch is up to date with 'origin/main'.

Changes to be committed:
  (use "git restore --staged <file>..." to unstage)
  modified:   .gitattributes
  	new file:   config.json
	new file:   sentencepiece.bpe.model
	new file:   special_tokens_map.json
	new file:   tf_model.h5
	new file:   tokenizer.json
	new file:   tokenizer_config.json
```
{/if}

Da mesma forma, podemos ter certeza de que o git-lfs está rastreando os arquivos corretos, utilizando seu comando `status`:

```bash
git lfs status
```

{#if fw === 'pt'}
```bash
On branch main
Objects to be pushed to origin/main:


Objects to be committed:

	config.json (Git: bc20ff2)
	pytorch_model.bin (LFS: 35686c2)
	sentencepiece.bpe.model (LFS: 988bc5a)
	special_tokens_map.json (Git: cb23931)
	tokenizer.json (Git: 851ff3e)
	tokenizer_config.json (Git: f0f7783)

Objects not staged for commit:


```

Podemos ver que todos os arquivos têm `Git` como manipulador, exceto *pytorch_model.bin* e *sentencepiece.bpe.model*, que têm `LFS`. Ótimo!

{:else}
```bash
On branch main
Objects to be pushed to origin/main:


Objects to be committed:

	config.json (Git: bc20ff2)
	sentencepiece.bpe.model (LFS: 988bc5a)
	special_tokens_map.json (Git: cb23931)
	tf_model.h5 (LFS: 86fce29)
	tokenizer.json (Git: 851ff3e)
	tokenizer_config.json (Git: f0f7783)

Objects not staged for commit:


```

Podemos ver que todos os arquivos têm `Git` como manipulador, exceto *t5_model.h5*, que tem `LFS`. Excelente!

{/if}

Vamos prosseguir para as etapas finais, comprometendo-nos e empurrando para o repositório remoto *huggingface.co*:

```bash
git commit -m "First model version"
```

{#if fw === 'pt'}
```bash
[main b08aab1] First model version
 7 files changed, 29027 insertions(+)
  6 files changed, 36 insertions(+)
 create mode 100644 config.json
 create mode 100644 pytorch_model.bin
 create mode 100644 sentencepiece.bpe.model
 create mode 100644 special_tokens_map.json
 create mode 100644 tokenizer.json
 create mode 100644 tokenizer_config.json
```
{:else}
```bash
[main b08aab1] First model version
 6 files changed, 36 insertions(+)
 create mode 100644 config.json
 create mode 100644 sentencepiece.bpe.model
 create mode 100644 special_tokens_map.json
 create mode 100644 tf_model.h5
 create mode 100644 tokenizer.json
 create mode 100644 tokenizer_config.json
```
{/if}

O push pode levar um pouco de tempo, dependendo da velocidade de sua conexão à Internet e do tamanho de seus arquivos:

```bash
git push
```

```bash
Uploading LFS objects: 100% (1/1), 433 MB | 1.3 MB/s, done.
Enumerating objects: 11, done.
Counting objects: 100% (11/11), done.
Delta compression using up to 12 threads
Compressing objects: 100% (9/9), done.
Writing objects: 100% (9/9), 288.27 KiB | 6.27 MiB/s, done.
Total 9 (delta 1), reused 0 (delta 0), pack-reused 0
To https://huggingface.co/lysandre/dummy
   891b41d..b08aab1  main -> main
```

{#if fw === 'pt'}
Se dermos uma olhada no repositório modelo quando este estiver terminado, podemos ver todos os arquivos recentemente adicionados:

<div class="flex justify-center">
<img src="https://huggingface.co/datasets/huggingface-course/documentation-images/resolve/main/en/chapter4/full_model.png" alt="The 'Files and versions' tab now contains all the recently uploaded files." width="80%"/>
</div>

A IU permite que você explore os arquivos modelo e os commits e veja as diferenças introduzidas por cada commit:

<div class="flex justify-center">
<img src="https://huggingface.co/datasets/huggingface-course/documentation-images/resolve/main/en/chapter4/diffs.gif" alt="The diff introduced by the recent commit." width="80%"/>
</div>
{:else}
Se dermos uma olhada no repositório modelo quando este estiver terminado, podemos ver todos os arquivos recentemente adicionados:

<div class="flex justify-center">
<img src="https://huggingface.co/datasets/huggingface-course/documentation-images/resolve/main/en/chapter4/full_model_tf.png" alt="The 'Files and versions' tab now contains all the recently uploaded files." width="80%"/>
</div>

A IU permite que você explore os arquivos modelo e os commits e veja as diferenças introduzidas por cada commit:

<div class="flex justify-center">
<img src="https://huggingface.co/datasets/huggingface-course/documentation-images/resolve/main/en/chapter4/diffstf.gif" alt="The diff introduced by the recent commit." width="80%"/>
</div>
{/if}


<EditOnGithub source="https://github.com/huggingface/course/blob/main/chapters/pt/chapter4/3.mdx" />