import json

def filter_json_data(file_a, file_b, output_file):
    """
    根据A文件（jsonl）的text和image字段过滤B文件（json）
    
    匹配条件：
    1. A文件的image字段与B文件的image字段完全一致
    2. A文件的text字段完全出现在B文件conversations中human的value字段中
    
    Args:
        file_a: A文件路径（jsonl格式，一行一行）
        file_b: B文件路径（json格式，有缩进）
        output_file: 输出文件路径
    """
    
    print("正在读取A文件（jsonl格式）...")
    # 读取A文件，创建text-image组合集合
    criteria = set()
    line_count = 0
    
    with open(file_a, 'r', encoding='utf-8') as f:
        for line in f:
            line = line.strip()
            if line:
                line_count += 1
                try:
                    data = json.loads(line)
                    text = data.get('text', '')
                    image = data.get('image', '')
                    if text and image:
                        criteria.add((text, image))
                except json.JSONDecodeError:
                    print(f"警告：A文件第{line_count}行JSON解析失败，跳过")
    
    print(f"A文件共读取 {line_count} 行，提取到 {len(criteria)} 个text-image组合")
    
    print("正在读取B文件（json格式）...")
    # 读取B文件
    try:
        with open(file_b, 'r', encoding='utf-8') as f:
            b_data = json.load(f)
    except json.JSONDecodeError as e:
        print(f"错误：B文件JSON解析失败 - {e}")
        return
    except FileNotFoundError:
        print(f"错误：找不到文件 {file_b}")
        return
    
    if not isinstance(b_data, list):
        print("错误：B文件应该包含一个JSON数组")
        return
    
    print(f"B文件包含 {len(b_data)} 条数据")
    
    # 过滤数据
    filtered_data = []
    removed_count = 0
    
    for i, item in enumerate(b_data):
        should_keep = True
        
        if 'conversations' in item and isinstance(item['conversations'], list):
            item_image = item.get('image', '')
            
            for conv in item['conversations']:
                if isinstance(conv, dict) and conv.get('from') == 'human':
                    human_value = conv.get('value', '')
                    
                    # 检查每个A文件中的text-image组合
                    for a_text, a_image in criteria:
                        # 条件1：image字段完全一致
                        # 条件2：text字段完全出现在human的value中
                        if (a_image == item_image and a_text in human_value):
                            should_keep = False
                            print(f"找到匹配项（第{i+1}条数据）：")
                            print(f"  A文件text: '{a_text[:100]}...'")
                            print(f"  A文件image: '{a_image}'")
                            print(f"  B文件image: '{item_image}'")
                            print(f"  B文件human value: '{human_value[:100]}...'")
                            print("  ---")
                            break
                    
                    if not should_keep:
                        break
        
        if should_keep:
            filtered_data.append(item)
        else:
            removed_count += 1
    
    print(f"删除了 {removed_count} 条匹配的数据")
    print(f"保留了 {len(filtered_data)} 条数据")
    
    print(f"正在保存到 {output_file}...")
    # 保存结果
    with open(output_file, 'w', encoding='utf-8') as f:
        json.dump(filtered_data, f, ensure_ascii=False, indent=2)
    
    print("处理完成！")

# 使用示例
if __name__ == "__main__":
    filter_json_data(
        '/home/bingxing2/ailab/scx6mh7/jkl/LLaVA_8_8_null_space/capture_know/sample_data/scienceqa/rank128_sample32.jsonl',
        '/home/bingxing2/ailab/group/ai4bio/renyuchen/jkl/scienceqa/llava_test_CQM-A.json',
        '/home/bingxing2/ailab/scx6mh7/jkl/LLaVA_8_8_null_space/capture_know/tichu_sample_data/new_tichu_32_scienceqa.jsonl'
    )