﻿param (
    [string]$InstallPath = $PSScriptRoot,
    [switch]$DisableProxy,
    [string]$UseCustomProxy,
    [switch]$DisablePyPIMirror,
    [switch]$DisableUV,
    [switch]$DisableGithubMirror,
    [string]$UseCustomGithubMirror,
    [string]$InstallBranch,
    [Parameter(ValueFromRemainingArguments=$true)]$ExtraArgs
)
$FOOOCUS_INSTALLER_VERSION = 190



# 消息输出
function Print-Msg ($msg) {
    Write-Host "[$(Get-Date -Format "yyyy-MM-dd HH:mm:ss")]" -ForegroundColor Yellow -NoNewline
    Write-Host "[Fooocus Installer]" -ForegroundColor Cyan -NoNewline
    Write-Host ":: " -ForegroundColor Blue -NoNewline
    Write-Host "$msg"
}


# 显示 Fooocus Installer 版本
function Get-Fooocus-Installer-Version {
    $ver = $([string]$FOOOCUS_INSTALLER_VERSION).ToCharArray()
    $major = ($ver[0..($ver.Length - 3)])
    $minor = $ver[-2]
    $micro = $ver[-1]
    Print-Msg "Fooocus Installer 版本: v${major}.${minor}.${micro}"
}


# 代理配置
function Set-Proxy {
    $Env:NO_PROXY = "localhost,127.0.0.1,::1"
    # 检测是否禁用自动设置镜像源
    if ((Test-Path "$PSScriptRoot/disable_proxy.txt") -or ($DisableProxy)) {
        Print-Msg "检测到本地存在 disable_proxy.txt 代理配置文件 / -DisableProxy 命令行参数, 禁用自动设置代理"
        return
    }

    $internet_setting = Get-ItemProperty -Path "HKCU:\Software\Microsoft\Windows\CurrentVersion\Internet Settings"
    if ((Test-Path "$PSScriptRoot/proxy.txt") -or ($UseCustomProxy)) { # 本地存在代理配置
        if ($UseCustomProxy) {
            $proxy_value = $UseCustomProxy
        } else {
            $proxy_value = Get-Content "$PSScriptRoot/proxy.txt"
        }
        $Env:HTTP_PROXY = $proxy_value
        $Env:HTTPS_PROXY = $proxy_value
        Print-Msg "检测到本地存在 proxy.txt 代理配置文件 / -UseCustomProxy 命令行参数, 已读取代理配置文件并设置代理"
    } elseif ($internet_setting.ProxyEnable -eq 1) { # 系统已设置代理
        $proxy_addr = $($internet_setting.ProxyServer)
        # 提取代理地址
        if (($proxy_addr -match "http=(.*?);") -or ($proxy_addr -match "https=(.*?);")) {
            $proxy_value = $matches[1]
            # 去除 http / https 前缀
            $proxy_value = $proxy_value.ToString().Replace("http://", "").Replace("https://", "")
            $proxy_value = "http://${proxy_value}"
        } elseif ($proxy_addr -match "socks=(.*)") {
            $proxy_value = $matches[1]
            # 去除 socks 前缀
            $proxy_value = $proxy_value.ToString().Replace("http://", "").Replace("https://", "")
            $proxy_value = "socks://${proxy_value}"
        } else {
            $proxy_value = "http://${proxy_addr}"
        }
        $Env:HTTP_PROXY = $proxy_value
        $Env:HTTPS_PROXY = $proxy_value
        Print-Msg "检测到系统设置了代理, 已读取系统中的代理配置并设置代理"
    }
}


# 下载 Fooocus Installer
function Download-Fooocus-Installer {
    # 可用的下载源
    $urls = @(
        "https://github.com/licyk/sd-webui-all-in-one/raw/main/fooocus_installer.ps1",
        "https://gitee.com/licyk/sd-webui-all-in-one/raw/main/fooocus_installer.ps1",
        "https://github.com/licyk/sd-webui-all-in-one/releases/download/fooocus_installer/fooocus_installer.ps1",
        "https://gitee.com/licyk/sd-webui-all-in-one/releases/download/fooocus_installer/fooocus_installer.ps1",
        "https://gitlab.com/licyk/sd-webui-all-in-one/-/raw/main/fooocus_installer.ps1"
    )
    $i = 0

    New-Item -ItemType Directory -Path "$PSScriptRoot/cache" -Force > $null

    ForEach ($url in $urls) {
        Print-Msg "正在下载最新的 Fooocus Installer 脚本"
        Invoke-WebRequest -Uri $url -OutFile "$PSScriptRoot/cache/fooocus_installer.ps1"
        if ($?) {
            Print-Msg "下载 Fooocus Installer 脚本成功"
            break
        } else {
            Print-Msg "下载 Fooocus Installer 脚本失败"
            $i += 1
            if ($i -lt $urls.Length) {
                Print-Msg "重试下载 Fooocus Installer 脚本"
            } else {
                Print-Msg "下载 Fooocus Installer 脚本失败, 可尝试重新运行 Fooocus Installer 下载脚本"
                return $false
            }
        }
    }
    return $true
}


# 获取本地配置文件参数
function Get-Local-Setting {
    $arg = @{}
    if ((Test-Path "$PSScriptRoot/disable_pypi_mirror.txt") -or ($DisablePyPIMirror)) {
        $arg.Add("-DisablePyPIMirror", $true)
    }

    if ((Test-Path "$PSScriptRoot/disable_proxy.txt") -or ($DisableProxy)) {
        $arg.Add("-DisableProxy", $true)
    } else {
        if ((Test-Path "$PSScriptRoot/proxy.txt") -or ($UseCustomProxy)) {
            if ($UseCustomProxy) {
                $proxy_value = $UseCustomProxy
            } else {
                $proxy_value = Get-Content "$PSScriptRoot/proxy.txt"
            }
            $arg.Add("-UseCustomProxy", $proxy_value)
        }
    }

    if ((Test-Path "$PSScriptRoot/disable_uv.txt") -or ($DisableUV)) {
        $arg.Add("-DisableUV", $true)
    }

    if ((Test-Path "$PSScriptRoot/disable_gh_mirror.txt") -or ($DisableGithubMirror)) {
        $arg.Add("-DisableGithubMirror", $true)
    } else {
        if ((Test-Path "$PSScriptRoot/gh_mirror.txt") -or ($UseCustomGithubMirror)) {
            if ($UseCustomGithubMirror) {
                $github_mirror = $UseCustomGithubMirror
            } else {
                $github_mirror = Get-Content "$PSScriptRoot/gh_mirror.txt"
            }
            $arg.Add("-UseCustomGithubMirror", $github_mirror)
        }
    }

    if ((Get-Command git -ErrorAction SilentlyContinue) -and (Test-Path "$PSScriptRoot/Fooocus/.git")) {
        $git_remote = $(git -C "$PSScriptRoot/Fooocus" remote get-url origin)
        $array = $git_remote -split "/"
        $branch = "$($array[-2])/$($array[-1])"
        if (($branch -eq "lllyasviel/Fooocus") -or ($branch -eq "lllyasviel/Fooocus.git")) {
            $arg.Add("-InstallBranch", "fooocus")
        } elseif (($branch -eq "MoonRide303/Fooocus-MRE") -or ($branch -eq "MoonRide303/Fooocus-MRE.git")) {
            $arg.Add("-InstallBranch", "fooocus_mre")
        } elseif (($branch -eq "runew0lf/RuinedFooocus") -or ($branch -eq "runew0lf/RuinedFooocus.git")) {
            $arg.Add("-InstallBranch", "ruined_fooocus")
        }
    } elseif ((Test-Path "$PSScriptRoot/install_fooocus.txt") -or ($InstallBranch -eq "fooocus")) {
        $arg.Add("-InstallBranch", "fooocus")
    } elseif ((Test-Path "$PSScriptRoot/install_fooocus_mre.txt") -or ($InstallBranch -eq "fooocus_mre")) {
        $arg.Add("-InstallBranch", "fooocus_mre")
    } elseif ((Test-Path "$PSScriptRoot/install_ruined_fooocus.txt") -or ($InstallBranch -eq "ruined_fooocus")) {
        $arg.Add("-InstallBranch", "ruined_fooocus")
    }

    $arg.Add("-InstallPath", $InstallPath)

    return $arg
}


# 处理额外命令行参数
function Get-ExtraArgs {
    $extra_args = New-Object System.Collections.ArrayList

    ForEach ($a in $ExtraArgs) {
        $extra_args.Add($a) | Out-Null
    }

    $params = $extra_args.ForEach{ 
        if ($_ -match '\s|"') { "'{0}'" -f ($_ -replace "'", "''") } 
        else { $_ } 
    } -join ' '

    return $params
}


function Main {
    Print-Msg "初始化中"
    Get-Fooocus-Installer-Version
    Set-Proxy

    $status = Download-Fooocus-Installer

    if ($status) {
        Print-Msg "运行 Fooocus Installer 中"
        $arg = Get-Local-Setting
        $extra_args = Get-ExtraArgs
        try {
            Invoke-Expression "& `"$PSScriptRoot/cache/fooocus_installer.ps1`" $extra_args @arg"
        }
        catch {
            Print-Msg "运行 Fooocus Installer 时出现了错误: $_"
            Read-Host | Out-Null
        }
    } else {
        Read-Host | Out-Null
    }
}

###################

Main
